import React, { useCallback, useEffect, useMemo } from 'react';

import { Loader } from '@yandex-cloud/uikit';
import { RouteComponentProps } from 'react-router';

import { PageTemplate, Tab, Tabs } from '../../../../../components/yp';
import { YpLocation } from '../../../../../models/api';
import { YpEntityRouteProps } from '../../../../../models/ui/yp/view';
import { useNetworkErrors, useYpPods, useYpPodSet } from '../../../../../redux';
import { getNavigation, getYPError } from '../../../utils';
import { networkRequests } from './models';
import { PodSetInfo } from './PodSetInfo/PodSetInfo';
import { PodSetPods } from './PodSetPods/PodSetPods';

import classes from './PodSet.module.css';

export const PodSet: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({ match }) => {
   const {
      params: { cluster, entityId },
   } = match;

   const { podSet } = useYpPodSet(entityId, cluster as YpLocation, networkRequests.podSet);
   const podsFilter = useMemo(
      () => ({
         podSetId: entityId,
      }),
      [entityId],
   );
   const navigation = useMemo(() => getNavigation(cluster, { id: 'pod-sets', title: 'pod sets' }, entityId), [
      cluster,
      entityId,
   ]);
   const { canFetch, isFetching, pods, requestPods } = useYpPods(
      cluster as YpLocation,
      networkRequests.pods,
      podsFilter,
   );
   const errors = useNetworkErrors(Object.values(networkRequests));
   useEffect(() => {
      requestPods(podsFilter, 1, true);
   }, []); //eslint-disable-line

   const podSetError = useMemo(() => getYPError(errors[networkRequests.podSet]), [errors]);

   const podsError = useMemo(() => getYPError(errors[networkRequests.pods]), [errors]);

   const onLoadMore = useCallback(
      newPage => {
         requestPods(
            {
               podSetId: entityId,
            },
            newPage,
            false,
         );
      },
      [requestPods, entityId],
   );

   if (!podSet && !Object.values(errors).length) {
      return (
         <div className={classes.loaderContainer}>
            <span className={classes.loader}>
               <Loader size={'s'} />
            </span>
            <span className={classes.loaderGap} />
         </div>
      );
   }

   const { meta, labels, status, spec } = podSet ?? {};

   const showPods = pods || podsError;
   const tabs: Tab[] = [];
   if (podSet) {
      tabs.push(
         {
            id: 'info',
            title: 'General',
            content: (
               <div className={classes.infoTab}>
                  <div className={classes.generalInfo}>
                     <PodSetInfo podSet={podSet} cluster={cluster} />
                  </div>
               </div>
            ),
         },
         { id: 'meta', title: 'Meta', obj: meta },
         { id: 'labels', title: 'Labels', obj: labels },
         { id: 'spec', title: 'Spec', obj: spec },
         { id: 'status', title: 'Status', obj: status },
      );

      if (showPods) {
         tabs.push({
            id: 'pods',
            title: <span>Pods {pods && <span className={classes.podsCount}>{pods.length}</span>}</span>,
            content: (
               <div>
                  {podsError ? (
                     <div className={classes.podsError}>{podsError}</div>
                  ) : (
                     <PodSetPods
                        cluster={cluster as YpLocation}
                        pods={pods}
                        canFetch={canFetch}
                        isFetching={isFetching}
                        onLoadMore={onLoadMore}
                     />
                  )}
               </div>
            ),
            width: 'full',
         });
      }
   }

   return (
      <PageTemplate error={podSetError} navigation={navigation}>
         {!podSet ? null : <Tabs tabs={tabs} />}
      </PageTemplate>
   );
};

PodSet.displayName = 'PodSet';
