import { Button, Loader } from '@yandex-cloud/uikit';
import React, { useCallback, useEffect, useMemo, useState } from 'react';
import { RouteComponentProps, useHistory } from 'react-router';

import { PageTemplate, PodsTable } from '../../../../components/yp';
import { YpLocation } from '../../../../models/api';
import { PodsFiltersParams, YpEntityRouteProps } from '../../../../models/ui/yp/view';
import { useNetworkErrors, useYpPods } from '../../../../redux';
import { setUrlQuery } from '../../../../utils';
import { getNavigation, getYPError } from '../../utils';

import classes from './Pods.module.css';
import { PodsFilters } from './PodsFilters/PodsFilters';
import { parseFiltersValuesFromSearch } from './utils';

const requestKey = 'ypClusterPodsRequestKey';

export const Pods: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({
   match: {
      params: { cluster },
   },
   location,
}) => {
   const [page, setPage] = useState(1);
   const [filters, setFilters] = useState<PodsFiltersParams>({});
   const history = useHistory();
   const { pods, requestPods, isFetching, canFetch } = useYpPods(cluster as YpLocation, requestKey);
   const errors = useNetworkErrors([requestKey]);
   const navigation = useMemo(() => getNavigation(cluster, 'pods'), [cluster]);

   useEffect(() => {
      if (!location.search) {
         requestPods(filters, page, true);
      } else {
         const filtersFromSearch = parseFiltersValuesFromSearch(location.search);

         setFilters(filtersFromSearch);
         setPage(1);
         requestPods(filtersFromSearch, 1, true);
      }
   }, []); //eslint-disable-line

   const onFiltersSubmit = useCallback(
      (newFilters: PodsFiltersParams) => {
         setUrlQuery(history, location, newFilters, false);
         setFilters(newFilters);
         setPage(1);
         requestPods(newFilters, 1, true);
      },
      [history, location, requestPods],
   );

   // TODO: выкинуть эти страницы в DEPLOY-4616
   const fetchMorePods = useCallback(() => {
      if (!isFetching) {
         requestPods(filters, page + 1, false);

         setPage(page + 1);
      }
   }, [page, isFetching, requestPods, filters]);

   const errorItem = useMemo(() => getYPError(errors[requestKey]), [errors]);

   return (
      <PageTemplate error={errorItem} navigation={navigation}>
         <div>
            <PodsFilters onSubmit={onFiltersSubmit} filters={filters} disabled={isFetching} />{' '}
         </div>
         {pods && (
            <div>
               <PodsTable
                  pods={pods}
                  cluster={cluster as YpLocation}
                  view={'cluster'}
                  emptyText={isFetching ? 'Loading pods...' : undefined}
               />
            </div>
         )}
         <div className={classes.loaderContainer}>
            {isFetching && <Loader size={'s'} />}
            {canFetch && !isFetching && (
               <Button
                  onClick={() => {
                     fetchMorePods();
                  }}
                  view={'outlined'}
               >
                  Load more
               </Button>
            )}
         </div>
      </PageTemplate>
   );
};
