import { TextInput } from '@yandex-cloud/uikit';
import { YCSelect } from '@yandex-data-ui/common';
import { classNames, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useState } from 'react';

import { LegoButton } from '../../../../../_lego';
import { EEvictionState, EPodCurrentState, ESchedulingState } from '../../../../../proto-typings';
import { PodsFiltersParams } from '../models';

import classes from './PodsFilters.module.css';

const stateSelectorItems = Object.values(EPodCurrentState).map(key => ({
   title: forHumanCapitalized(key),
   value: key,
}));

const evictionSelectorItems = Object.values(EEvictionState).map(key => ({
   title: forHumanCapitalized(key),
   value: key,
}));

const schedulingSelectorItems = Object.values(ESchedulingState).map(key => ({
   title: forHumanCapitalized(key),
   value: key,
}));

interface Props {
   filters: PodsFiltersParams;
   disabled: boolean;
   className?: string;

   onSubmit(params: PodsFiltersParams): void;
}

export const PodsFilters: React.FC<Props> = ({ onSubmit, filters, disabled, className }) => {
   const [filtersState, setFiltersState] = useState<PodsFiltersParams>(filters);

   useEffect(() => {
      setFiltersState(filters);
   }, [filters]);

   const onInputChange = useCallback(
      <T extends keyof Required<PodsFiltersParams>>(fieldName: T, fieldValue: Required<PodsFiltersParams[T]>) => {
         const newFiltersState = { ...filtersState, [fieldName]: fieldValue };
         setFiltersState(newFiltersState);
      },
      [filtersState],
   );

   return (
      <div className={classNames(classes.filters, className)}>
         <div className={classes.row}>
            <TextInput
               value={filtersState.podId || ''}
               onUpdate={value => {
                  onInputChange('podId', value);
               }}
               placeholder={'Pod ID'}
               disabled={disabled}
            />
            <TextInput
               value={filtersState.podSetId || ''}
               onUpdate={value => {
                  onInputChange('podSetId', value);
               }}
               placeholder={'Pod set ID'}
               disabled={disabled}
            />
            <TextInput
               value={filtersState.service || ''}
               onUpdate={value => {
                  onInputChange('service', value);
               }}
               placeholder={'Service/stage ID'}
               disabled={disabled}
            />
            <YCSelect
               label={'State: '}
               type={YCSelect.MULTIPLE}
               items={stateSelectorItems}
               value={filtersState.podState}
               onUpdate={value => {
                  onInputChange('podState', value as EPodCurrentState[]);
               }}
               disabled={disabled}
            />
            <YCSelect
               label={'Eviction: '}
               type={YCSelect.MULTIPLE}
               items={evictionSelectorItems}
               value={filtersState.evictionState}
               onUpdate={value => {
                  onInputChange('evictionState', value as EEvictionState[]);
               }}
               disabled={disabled}
            />
            <YCSelect
               label={'Scheduling: '}
               type={YCSelect.MULTIPLE}
               items={schedulingSelectorItems}
               value={filtersState.schedulingState}
               onUpdate={value => {
                  onInputChange('schedulingState', value as ESchedulingState[]);
               }}
               disabled={disabled}
            />
            <LegoButton
               onClick={() => {
                  onSubmit({});
               }}
               disabled={disabled}
            >
               Reset
            </LegoButton>
            <LegoButton
               onClick={() => {
                  onSubmit({ ...filtersState });
               }}
               disabled={disabled}
            >
               Search
            </LegoButton>
         </div>
         <div className={classes.row}>
            <TextInput
               value={filtersState.query}
               onUpdate={value => {
                  onInputChange('query', value);
               }}
               placeholder={'Query example: [/spec/resource_requests/vcpu_guarantee] > 10000'}
               disabled={disabled}
            />
         </div>
      </div>
   );
};
