import { Button } from '@yandex-cloud/uikit';

import {
   classNames,
   Json,
   modalService,
   TIMES_IN_MS,
   UpdateTimerMode,
   useDismounted,
   useUpdateTimer,
} from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { match } from 'react-router';
import { takeUntil } from 'rxjs/operators';
import format from 'unipika';
import 'unipika/dist/unipika.css';
import { PageTemplate, Tab, Tabs } from '../../../../../components/yp';
import { DeepPartial } from '../../../../../models';

import { YpLocation } from '../../../../../models/api';
import { YpEntityRouteProps } from '../../../../../models/ui/yp/view';
import { EEvictionState, TPod, TPodStatus } from '../../../../../proto-typings';
import { getPodInfo, RootState, selectIsPodExists, useNetworkErrors, useYpNode, useYpPod } from '../../../../../redux';
import { ypApi } from '../../../../../services';
import { getNavigation, getYPError } from '../../../utils';
import { PodEvictionPopup } from './__evictionPopup/PodEvictionPopup';
import { PodAction, podErrorDefaultText, podRequests } from './models';

import classes from './Pod.module.css';
import { PodHistory } from './PodHistory/PodHistory';
import { PodInfo } from './PodInfo/PodInfo';
import { getStatusWithAnnotatedValues } from './utils';

const unipikaFormat = format();
function getUnipikaHtml(obj: any) {
   try {
      return unipikaFormat.formatFromYSON(obj, { indent: 2 });
   } catch (e) {
      return `Parsing error: ${e}`;
   }
}
const UnipikaJson: React.FC<{ obj: any }> = ({ obj }) => {
   const html = getUnipikaHtml(obj);
   return (
      <pre
         className={classNames('unipika', classes.unipika)}
         // eslint-disable-next-line react/no-danger
         dangerouslySetInnerHTML={{
            __html: html,
         }}
      />
   );
};

type Props = {
   match: match<YpEntityRouteProps>;
};

export const Pod: React.FC<Props> = ({ match: routeMatch }) => {
   const {
      params: { cluster, entityId },
   } = routeMatch;

   const { pod, timestamp } = useYpPod(entityId, cluster as YpLocation, podRequests.pod);
   const node = useYpNode(pod?.spec?.node_id, cluster as YpLocation, podRequests.node);
   const navigation = useMemo(() => getNavigation(cluster, 'pods', entityId), [cluster, entityId]);
   const isPodExists = useSelector((state: RootState) => selectIsPodExists(state, entityId));

   const dispatch = useDispatch();
   const podUpdateCallback = useCallback(() => {
      dispatch(getPodInfo.withRequestKey(podRequests.pod)(entityId, cluster as YpLocation));
   }, [dispatch, entityId, cluster]);

   const [podAnnotated, setPodAnnotated] = useState<{ pod: DeepPartial<TPod> }>();
   const podAnnotatedCallback = useCallback(() => ypApi.getPodAnnotated(entityId, cluster as YpLocation), [
      entityId,
      cluster,
   ]);

   const errors = useNetworkErrors(Object.keys(podErrorDefaultText));
   const errorItems = useMemo(
      () =>
         Object.keys(podErrorDefaultText)
            .map(errorKey => getYPError(errors[errorKey]))
            .filter(Boolean),
      [errors],
   );

   useUpdateTimer({
      callback: podUpdateCallback,
      fast: TIMES_IN_MS.Second * 10,
      slow: TIMES_IN_MS.Minute,
      mode: UpdateTimerMode.Slow,
   });

   const dismounted = useDismounted();
   useEffect(() => {
      podAnnotatedCallback()
         .pipe(takeUntil(dismounted))
         .subscribe(result => {
            setPodAnnotated(result);
         });
   }, [dismounted, podAnnotatedCallback]);

   const showModal = useCallback(
      (action: PodAction) => {
         if (timestamp) {
            modalService
               .open(PodEvictionPopup, {
                  action,
                  timestamp,
                  location: cluster as YpLocation,
                  podId: entityId,
               })
               .pipe(takeUntil(dismounted))
               .subscribe(() => {
                  podUpdateCallback();
               });
         }
      },
      [timestamp, cluster, entityId, dismounted, podUpdateCallback],
   );

   const evictionAction = useCallback(
      (action: PodAction) => {
         if (!pod?.spec?.node_id) {
            return;
         }

         const eviction = pod?.status?.eviction;
         if (!eviction) {
            return;
         }
         showModal(action);
         dispatch(getPodInfo.withRequestKey(podRequests.pod)(entityId, cluster as YpLocation));
      },
      [pod, entityId, dispatch, cluster, showModal],
   );

   const { labels, meta, spec, status } = pod || {};

   const statusWithAnnotated: TPodStatus | undefined = useMemo(
      () =>
         getStatusWithAnnotatedValues(status, podAnnotated?.pod?.status, [
            'pod_dynamic_attributes.annotations',
            'agent.current_spec_timestamp',
         ]),
      [podAnnotated?.pod?.status, status],
   );

   const tabs: Tab[] = ([
      pod && {
         id: 'info',
         title: 'General',
         content: (
            <div>
               <PodInfo node={node} pod={pod} cluster={cluster} />
            </div>
         ),
      },
      meta && {
         id: 'meta',
         title: 'Meta',
         content: <Json obj={meta} />,
      },
      labels && {
         id: 'labels',
         title: 'Labels',
         content: <Json obj={labels} />,
      },
      spec && {
         id: 'spec',
         title: 'Spec',
         content: <Json obj={spec} />,
      },
      status && {
         id: 'status',
         title: 'Status',
         content: <UnipikaJson obj={statusWithAnnotated} />,
      },
      isPodExists !== undefined
         ? {
              id: 'history',
              title: 'History',
              content: (
                 <PodHistory
                    podId={entityId}
                    uuid={pod?.meta?.uuid}
                    cluster={cluster as YpLocation}
                    requestKey={podRequests.podHistory}
                 />
              ),
           }
         : null,
   ] as (Tab | undefined)[]).filter((item): item is Tab => Boolean(item));

   const evictionState = status?.eviction?.state;
   const deployEngine = pod?.labels?.deploy_engine;
   let controls = null;

   if (
      spec?.node_id &&
      evictionState &&
      evictionState !== EEvictionState.ES_ACKNOWLEDGED &&
      (deployEngine === 'RSC' || deployEngine === 'MCRSC') // TODO: выпилить эту проверку после завершения тикета про доступы к подам в Няне
   ) {
      controls = (
         <div className={classes.controls}>
            {evictionState === EEvictionState.ES_NONE && (
               <Button
                  size={'s'}
                  view={'outlined-danger'}
                  onClick={() => {
                     evictionAction('evict');
                  }}
                  className={classes.evictionBtn}
               >
                  Request eviction
               </Button>
            )}
            {evictionState === EEvictionState.ES_REQUESTED && (
               <>
                  <Button
                     size={'s'}
                     view={'outlined-info'}
                     onClick={() => {
                        evictionAction('acknowledge');
                     }}
                     className={classes.evictionBtn}
                  >
                     Acknowledge eviction
                  </Button>
                  <Button
                     size={'s'}
                     view={'outlined-danger'}
                     onClick={() => {
                        evictionAction('abort');
                     }}
                     className={classes.evictionBtn}
                  >
                     Abort eviction
                  </Button>
               </>
            )}
         </div>
      );
   }

   return (
      <PageTemplate navigation={navigation} controls={controls} error={errorItems.length ? errorItems : undefined}>
         <Tabs tabs={tabs} />
      </PageTemplate>
   );
};

Pod.displayName = 'Pod';
