import React from 'react';

import { CopyableText, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import { formatDistance } from 'date-fns';
import { Link } from 'react-router-dom';

import { StatusState } from '../../../../../../components/lib';
import {
   EEventType,
   TPodStatus_TEviction,
   TPodStatus_TMaintenance,
   TPodStatus_TScheduling,
} from '../../../../../../proto-typings';
import { urlBuilder } from '../../../../../../models';
import { HistoryEventRow, TimelineEvent } from '../../../../../../components/yp';
import { EvictionStateMap, MaintenanceStateMap, SchedulingStateMap } from '../../../../../../models/ui/yp/models';
import {
   isEvictionsEqual,
   isMaintenancesEqual,
   isSchedulingsEqual,
   PodHistoryEvent,
} from '../../../../../../models/ui/yp/view';

const getEvictionViewData = (
   eviction: TPodStatus_TEviction | null,
   prevEviction: TPodStatus_TEviction | null,
   eventType: EEventType,
): TimelineEvent => {
   const isEqual = isEvictionsEqual(eviction, prevEviction);
   let state;
   if (eventType === EEventType.ET_OBJECT_REMOVED) {
      state = StatusState.Inactive;
   } else if (eviction) {
      state = EvictionStateMap.get(eviction.state) ?? StatusState.Unknown;
   } else {
      state = StatusState.Inactive;
   }

   const title =
      eventType === EEventType.ET_OBJECT_REMOVED ? 'Pod removed' : forHumanCapitalized(eviction?.state ?? '');
   const isChanged = eventType === EEventType.ET_OBJECT_REMOVED ? true : !isEqual;

   const evictionContent: HistoryEventRow[] | undefined = eviction
      ? [
           {
              title: 'Reason:',
              content: forHumanCapitalized(eviction.reason),
              key: 'reason',
           },
        ]
      : undefined;

   return {
      isChanged,
      title,
      state,
      message: eviction?.message,
      content: evictionContent,
   };
};

const getMaintenanceViewData = (
   maintenance: TPodStatus_TMaintenance | null,
   prevMaintenance: TPodStatus_TMaintenance | null,
   eventType: EEventType,
): TimelineEvent => {
   const isEqual = isMaintenancesEqual(maintenance, prevMaintenance);
   let state;
   if (eventType === EEventType.ET_OBJECT_REMOVED) {
      state = StatusState.Inactive;
   } else if (maintenance) {
      state = MaintenanceStateMap.get(maintenance.state) ?? StatusState.Unknown;
   } else {
      state = StatusState.Inactive;
   }

   const maintenanceContent: HistoryEventRow[] = [];

   if (maintenance?.info) {
      const {
         message,
         disruptive,
         kind,
         node_set_id,
         priority,
         source,
         estimated_duration,
         id,
         uuid,
      } = maintenance.info;

      maintenanceContent.push(
         {
            title: 'Info',
            content: <span />,
            key: 'info',
         },
         {
            title: 'ID:',
            content: id && (
               <span>
                  {id}
                  <CopyableText text={id} />
               </span>
            ),
            key: 'id',
         },
         {
            title: 'UUID:',
            content: uuid && (
               <span>
                  {uuid}
                  <CopyableText text={uuid} />
               </span>
            ),
            key: 'uuid',
         },
         {
            title: 'Kind:',
            content: kind && forHumanCapitalized(kind),
            key: 'kind',
         },
         {
            title: 'Node set ID:',
            content: node_set_id && (
               <span>
                  {node_set_id}
                  <CopyableText text={node_set_id} />
               </span>
            ),
            key: 'node-set-id',
         },
         {
            title: 'Message:',
            content: message,
            key: 'message',
         },
         {
            title: 'Disruptive:',
            content: disruptive.toString(),
            key: 'disruptive',
         },
         {
            title: 'Estimated duration:',
            content:
               estimated_duration?.seconds && formatDistance(new Date(estimated_duration.seconds * 1000), new Date(0)),
            key: 'duration',
         },
         {
            title: 'Priority:',
            content: priority && forHumanCapitalized(priority),
            key: 'proprity',
         },
         {
            title: 'Source:',
            content: source && forHumanCapitalized(source),
            key: 'source',
         },
      );
   }
   const title =
      eventType === EEventType.ET_OBJECT_REMOVED ? 'Pod removed' : forHumanCapitalized(maintenance?.state ?? '');
   const isChanged = eventType === EEventType.ET_OBJECT_REMOVED ? true : !isEqual;

   return {
      isChanged,
      state,
      title,
      content: maintenanceContent,
      message: maintenance?.message,
   };
};

const getSchedulingViewData = (
   scheduling: TPodStatus_TScheduling | null,
   prevScheduling: TPodStatus_TScheduling | null,
   eventType: EEventType,
   cluster: string,
): TimelineEvent => {
   let state;
   if (eventType === EEventType.ET_OBJECT_REMOVED) {
      state = StatusState.Inactive;
   } else if (scheduling) {
      state = SchedulingStateMap.get(scheduling.state) ?? StatusState.Unknown;
   } else {
      state = StatusState.Inactive;
   }

   const isEqual = isSchedulingsEqual(scheduling, prevScheduling);
   const schedulingContent: HistoryEventRow[] = [
      {
         title: 'Node ID:',
         content: scheduling?.node_id && (
            <span>
               <Link to={urlBuilder.ypNode(cluster, scheduling.node_id)}>{scheduling.node_id}</Link>
               <CopyableText text={scheduling.node_id} />
            </span>
         ),
         key: 'node-id',
      },
   ];

   const title =
      eventType === EEventType.ET_OBJECT_REMOVED ? 'Pod removed' : forHumanCapitalized(scheduling?.state ?? '');
   const isChanged = eventType === EEventType.ET_OBJECT_REMOVED ? true : !isEqual;

   return {
      isChanged,
      state,
      title,
      content: schedulingContent,
      message: scheduling?.message,
   };
};

export const getHistoryRows = (history: PodHistoryEvent[], cluster: string) =>
   history.map((item, index) => {
      const {
         result: {
            status: { eviction, maintenance, scheduling },
         },
         event_type,
         time: { seconds },
         user,
      } = item;

      const evictionEventData = getEvictionViewData(eviction, history[index + 1]?.result.status.eviction, event_type);
      const maintenanceEventData = getMaintenanceViewData(
         maintenance,
         history[index + 1]?.result.status.maintenance,
         event_type,
      );
      const schedulingEventData = getSchedulingViewData(
         scheduling,
         history[index + 1]?.result.status.scheduling,
         event_type,
         cluster,
      );

      return {
         timestamp: seconds,
         events: [evictionEventData, maintenanceEventData, schedulingEventData],
         user,
      };
   });
