import React, { ReactNode } from 'react';

import { ExternalLink, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import { Link } from 'react-router-dom';

import { EXTERNAL_LINKS, urlBuilder } from '../../../../../../models';
import { EEvictionState, EPodMaintenanceState, TNode, TPod } from '../../../../../../proto-typings';
import { DefinitionList, DefinitionListItem, wrapCopy, Status } from '../../../../../../components/lib';
import { FormatDate } from '../../../../../../components/yp/FormatDate/FormatDate';
import {
   getAddressAllocationItems,
   getAddressRequestItems,
   getDiskVolumeRequestItems,
   getGpuRequestsItems,
   getResourceRequestsFields,
   getSubnetRequestItems,
} from '../utils';

import { InfoGroup } from '../../../../../../components/yp/InfoGroup/InfoGroup';
import { Labels } from '../../../../../../components/yp/Labels/Labels';
import { StatusMap } from '../../../../../../models/ui/yp/models';

import classes from './PodInfo.module.css';

interface Props {
   pod: TPod;
   node?: TNode;
   cluster: string;
}

export const PodInfo: React.FC<Props> = ({ pod, node, cluster }) => {
   const { labels, status, spec, meta } = pod;
   const fields: DefinitionListItem[] = [
      {
         name: 'Pod ID',
         content: meta?.id,
         copyText: true,
      },
   ];

   const deployEngine = labels?.deploy_engine;
   if (deployEngine) {
      fields.push({
         name: 'Deploy engine',
         content: deployEngine === 'RSC' || deployEngine === 'MCRSC' ? `Deploy/${deployEngine}` : 'YP Lite',
      });
   }

   const stageId = labels?.deploy?.stage_id;
   if (stageId) {
      fields.push({
         name: 'Stage ID',
         content: wrapCopy(<Link to={urlBuilder.stage(stageId)}>{stageId}</Link>, stageId),
      });
   }

   const nannyServiceId = labels?.nanny_service_id;
   if (nannyServiceId) {
      const nannyServiceURL = labels && labels['yp.deploy_engine_url'];

      fields.push({
         name: 'Nanny service ID',
         content: wrapCopy(
            <ExternalLink
               className={classes.externalLink}
               href={nannyServiceURL || EXTERNAL_LINKS.nannyService(nannyServiceId)}
            >
               {nannyServiceId}
            </ExternalLink>,
            nannyServiceId,
         ),
      });
   }

   const nodeId = spec?.node_id;

   if (nodeId) {
      if (node) {
         fields.push({
            name: 'Node ID',
            content: <Link to={urlBuilder.ypNode(cluster, nodeId)}>{nodeId}</Link>,
            copyText: nodeId,
         });
      } else {
         fields.push({
            name: 'Node ID',
            content: nodeId,
            copyText: true,
         });
      }
   } else {
      fields.push({
         name: 'Node ID',
         content: 'Not allocated',
      });
   }

   if (meta?.pod_set_id) {
      fields.push({
         name: 'Pod set ID',
         content: wrapCopy(
            <Link to={urlBuilder.ypPodSet(cluster, meta.pod_set_id)}>{meta.pod_set_id}</Link>,
            meta.pod_set_id,
         ),
      });
   }

   fields.push(
      {
         name: 'Account ID',
         content: wrapCopy(meta?.account_id),
      },
      {
         name: 'Effective account ID',
         content: wrapCopy(meta?.effective_account_id),
      },
   );

   if (node) {
      fields.push({
         name: 'Datacenter',
         content: node.labels?.topology?.dc,
      });
   }

   const state = status?.agent?.state;
   if (state) {
      fields.push({
         name: 'Status',
         content: (
            <Status state={StatusMap.get(state)!} isAnimated={false}>
               {forHumanCapitalized(state)}
            </Status>
         ),
      });
   }

   const evictionState = status?.eviction?.state;
   if (evictionState && evictionState !== EEvictionState.ES_NONE) {
      fields.push({
         name: 'Eviction',
         content: (
            <span>
               {forHumanCapitalized(evictionState)} <FormatDate timestamp={status?.eviction?.last_updated} />
            </span>
         ),
      });
   }

   const maintenanceState = status?.maintenance?.state;
   if (maintenanceState && maintenanceState !== EPodMaintenanceState.PMS_NONE) {
      fields.push({
         name: 'Maintenance',
         content: (
            <span>
               {forHumanCapitalized(maintenanceState)} <FormatDate timestamp={status?.maintenance?.last_updated} />
            </span>
         ),
      });
   }
   fields.push(
      {
         name: 'Revision',
         content: spec?.pod_agent_payload?.spec?.revision,
      },
      {
         name: 'Last heartbeat time',
         content: <FormatDate timestamp={status?.agent?.last_heartbeat_time} empty={'Unknown'} />,
      },
      {
         name: 'Creation time',
         content: <FormatDate timestamp={meta?.creation_time} empty={'Unknown'} />,
      },
   );

   const addressResourcesContent: ReactNode[] = [];
   const addressRequests = spec?.ip6_address_requests;
   if (addressRequests) {
      addressResourcesContent.push(
         <InfoGroup title={'IP6 address requests'}>
            {addressRequests.map(request => {
               const items = getAddressRequestItems(request);

               return (
                  <InfoGroup>
                     <DefinitionList items={items} />
                  </InfoGroup>
               );
            })}
         </InfoGroup>,
      );
   }

   const ip6AddressAllocations = status?.ip6_address_allocations;
   if (ip6AddressAllocations) {
      addressResourcesContent.push(
         <InfoGroup title={'IP6 address allocations'}>
            {ip6AddressAllocations.map(request => {
               const items = getAddressAllocationItems(request);

               return (
                  <InfoGroup>
                     <DefinitionList items={items} />
                  </InfoGroup>
               );
            })}
         </InfoGroup>,
      );
   }

   const subnetRequests = spec?.ip6_subnet_requests;
   if (subnetRequests) {
      addressResourcesContent.push(
         <InfoGroup title={'Subnet requests'}>
            {subnetRequests.map(request => {
               const items = getSubnetRequestItems(request);

               return (
                  <InfoGroup>
                     <DefinitionList items={items} />
                  </InfoGroup>
               );
            })}
         </InfoGroup>,
      );
   }

   const resourcesContent: ReactNode[] = [];
   const resourceRequest = spec?.resource_requests;
   if (resourceRequest) {
      resourcesContent.push(
         <InfoGroup title={'Resource requests'}>
            <DefinitionList responsive={true} items={getResourceRequestsFields(resourceRequest)} />
         </InfoGroup>,
      );
   }

   const diskVolumeRequests = spec?.disk_volume_requests;
   if (diskVolumeRequests) {
      resourcesContent.push(
         <InfoGroup title={'Disk volumes'}>
            {diskVolumeRequests.map(request => (
               <InfoGroup>
                  <InfoGroup margin={'row'}>
                     <DefinitionList responsive={true} items={getDiskVolumeRequestItems(request)} />
                  </InfoGroup>
                  <Labels labels={request.labels} responsive={true} />
               </InfoGroup>
            ))}
         </InfoGroup>,
      );
   }

   const gpuRequests = spec?.gpu_requests;
   if (gpuRequests) {
      resourcesContent.push(
         <InfoGroup title={'GPUs'}>
            {gpuRequests.map(request => {
               const items = getGpuRequestsItems(request);

               return (
                  <InfoGroup>
                     <DefinitionList items={items} responsive={true} />
                  </InfoGroup>
               );
            })}
         </InfoGroup>,
      );
   }

   return (
      <div className={classes.container}>
         <div className={classes.info}>
            <DefinitionList items={fields} />
            <div className={classes.addressResources}>{addressResourcesContent}</div>
         </div>
         <div className={classes.resources}>
            <div className={classes.resourcesInner}>{resourcesContent}</div>
         </div>
      </div>
   );
};

PodInfo.displayName = 'Pod__PodInfo';
