import { Button } from '@yandex-cloud/uikit';
import { DISMISS_REASON, forHumanCapitalized, IModalProps, ModalLayout, TextField2 } from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import React, { PropsWithChildren, useCallback } from 'react';

import { useDispatch } from 'react-redux';
import { object, string } from 'yup';
import { YpLocation } from '../../../../../../models/api';
import { TicketAction } from '../../../../../../models/ui';
import { abortPodEviction, acknowledgePodEviction, requestPodEviction } from '../../../../../../redux';
import { PodAction, podRequests } from '../models';
import classes from './PodEvictionPopup.module.css';

const initialValue = {
   message: '',
};

export type TicketActionParams = typeof initialValue;

const ticketActionParamsValidationSchema = object<TicketActionParams>({
   message: string().required(),
});

const InternalForm: React.FC<PropsWithChildren<any> & { action: TicketAction }> = ({ children, action }) => (
   <Form className={classes.form}>
      <TextField2 name={'message'} label={'Message'} placeholder={'Enter message'} required={true} />
      {children}
   </Form>
);

interface Props extends IModalProps<TicketActionParams> {
   action: PodAction;
   podId: string;
   timestamp: number;
   location: YpLocation;
}

export const PodEvictionPopup: React.FC<Props> = React.memo(({ action, podId, timestamp, location, ok, cancel }) => {
   const dispatch = useDispatch();

   const handleSubmit = useCallback(
      (v: TicketActionParams, helpers: FormikHelpers<TicketActionParams>) => {
         const { message } = v;

         switch (action) {
            case 'evict':
               dispatch(
                  requestPodEviction.withRequestKey(podRequests.podEviction)(podId, timestamp, message, location),
               );
               break;
            case 'abort':
               dispatch(
                  abortPodEviction.withRequestKey(podRequests.podEvictionAbort)(podId, timestamp, message, location),
               );
               break;
            case 'acknowledge':
               dispatch(
                  acknowledgePodEviction.withRequestKey(podRequests.podEvictionAck)(
                     podId,
                     timestamp,
                     message,
                     location,
                  ),
               );
               break;
         }

         helpers.setSubmitting(false);
         ok(v);
      },
      [action, location, podId, timestamp, ok, dispatch],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const submitText = forHumanCapitalized(action);

   return (
      <ModalLayout title={'Enter message'} showFooter={false} onDismiss={dismiss} data-e2e={'Ticket:ActionModal'}>
         <Formik
            initialValues={initialValue}
            onSubmit={handleSubmit}
            validationSchema={ticketActionParamsValidationSchema}
         >
            {form => (
               <InternalForm action={action}>
                  <footer className={classes.footer}>
                     <Button
                        view={'normal'}
                        className={classes.cancelBtn}
                        onClick={dismiss}
                        disabled={form.isSubmitting}
                     >
                        Cancel
                     </Button>

                     <Button view={'action'} type={'submit'} disabled={!form.isValid || form.isSubmitting}>
                        {submitText}
                     </Button>
                  </footer>
               </InternalForm>
            )}
         </Formik>
      </ModalLayout>
   );
});

PodEvictionPopup.displayName = 'Pod__PodEvictionPopup';
