import { formatNumber } from '@yandex-infracloud-ui/libs';
import { get, mergeWith } from 'lodash-es';
import { DefinitionListItem, wrapCopy } from '../../../../../components/lib';
import { DeepPartial } from '../../../../../models';

import {
   TPodSpec_TDiskVolumeRequest,
   TPodSpec_TGpuRequest,
   TPodSpec_TIP6AddressRequest,
   TPodSpec_TIP6SubnetRequest,
   TPodSpec_TResourceRequests,
   TPodStatus,
   TPodStatus_TIP6AddressAllocation,
} from '../../../../../proto-typings';
import { ypMerge } from '../../../../../redux/slices/yp/model';
import { deepUpdateObject } from '../../../../../utils';
import { formatBytes, formatCores } from '../../../../../utils/yp';

type ResourceItemValue = number | string | boolean | undefined;

const getItemContent = (value: ResourceItemValue, format?: string, suffix = '') => {
   if (value === undefined) {
      return undefined;
   }

   if (typeof value === 'number' && format) {
      return `${formatNumber(value, format)}${suffix}`;
   }
   return `${value}${suffix}`;
};

export const getResourceRequestsFields = (resourceRequests: TPodSpec_TResourceRequests): DefinitionListItem[] => [
   {
      name: 'vCPU Limit',
      content: getItemContent(resourceRequests.vcpu_limit / 1000, formatCores, ' vcores'),
   },
   {
      name: 'vCPU Guarantee',
      content: getItemContent(resourceRequests.vcpu_guarantee / 1000, formatCores, ' vcores'),
   },
   {
      name: 'Memory Limit',
      content: getItemContent(resourceRequests.memory_limit, formatBytes),
   },
   {
      name: 'Memory Guarantee',
      content: getItemContent(resourceRequests.memory_guarantee, formatBytes),
   },
];

export const getAddressRequestItems = (request: TPodSpec_TIP6AddressRequest): DefinitionListItem[] => [
   {
      name: 'Network ID',
      content: wrapCopy(request.network_id),
   },
   {
      name: 'Vlan ID',
      content: request.vlan_id,
   },
   {
      name: 'Enable DNS',
      content: request.enable_dns,
   },
];

export const getAddressAllocationItems = (allocation: TPodStatus_TIP6AddressAllocation): DefinitionListItem[] => [
   {
      name: 'Vlan ID',
      content: allocation.vlan_id,
   },
   {
      name: 'Address',
      content: wrapCopy(allocation.address),
   },
   {
      name: 'Persistent FQDN',
      content: wrapCopy(allocation.persistent_fqdn),
   },
   {
      name: 'Transient FQDN',
      content: wrapCopy(allocation.transient_fqdn),
   },
];

export const getSubnetRequestItems = (request: TPodSpec_TIP6SubnetRequest): DefinitionListItem[] => [
   {
      name: 'Network ID',
      content: request.network_id,
   },
   {
      name: 'Vlan ID',
      content: request.vlan_id,
   },
];

export const getGpuRequestsItems = (request: TPodSpec_TGpuRequest): DefinitionListItem[] => [
   {
      name: 'Model',
      content: request.model,
   },
   {
      name: 'Max memory',
      content: getItemContent(request.max_memory, formatBytes),
   },
   {
      name: 'Min memory',
      content: getItemContent(request.min_memory, formatBytes),
   },
];

export const getDiskVolumeRequestItems = (request: TPodSpec_TDiskVolumeRequest): DefinitionListItem[] => [
   {
      name: 'ID',
      content: request.id,
      copyText: true,
   },
   {
      name: 'Storage provisioner',
      content: request.storage_provisioner,
   },
   {
      name: 'Storage class',
      content: request.storage_class,
   },
   {
      name: 'Capacity',
      content: getItemContent(request.quota_policy?.capacity, formatBytes),
   },
   {
      name: 'Bandwidth limit',
      content: getItemContent(request.quota_policy?.bandwidth_limit, formatBytes, '/s'),
   },
   {
      name: 'Bandwidth guarantee',
      content: getItemContent(request.quota_policy?.bandwidth_guarantee, formatBytes, '/s'),
   },
];

export const getStatusWithAnnotatedValues = (
   status: TPodStatus | undefined,
   podAnnotated: DeepPartial<TPodStatus> | undefined,
   paths: string[],
): TPodStatus | undefined => {
   if (!status) {
      return undefined;
   }
   let statusWithAnnotatedValues: TPodStatus | undefined;

   try {
      statusWithAnnotatedValues = JSON.parse(JSON.stringify(status));
      // eslint-disable-next-line no-empty
   } catch {}

   if (!statusWithAnnotatedValues) {
      return undefined;
   }

   if (!podAnnotated) {
      return statusWithAnnotatedValues;
   }

   try {
      // только нужные пути
      const annotatedPart = filterObjectByPaths(podAnnotated, paths);

      return mergeWith(statusWithAnnotatedValues, annotatedPart, (A, B) => {
         for (const key of Object.keys(B)) {
            if (key in A) {
               const newValue = B[key];
               if (typeof newValue === 'object' && newValue !== null && '$value' in newValue) {
                  const ypValue = getYpValue(newValue);
                  A[key] = ypValue;
                  B[key] = ypValue;
               }
            }
         }
         return ypMerge(A, B);
      });
   } catch (e) {
      return undefined;
   }
};

function filterObjectByPaths(object: object, paths: string[]): object {
   const result: object = {};

   for (const path of paths) {
      const value = get(object, path);

      if (value !== undefined) {
         deepUpdateObject(result, path.split('.'), value);
      }
   }

   return result;
}

type AnnotatedValue = {
   $value: string;
   $type: string;
};

type KnownAnnotatedTypes = 'uint64';
const parseAnnotated: Record<KnownAnnotatedTypes, (value: AnnotatedValue) => any> = {
   uint64: value => value,
};
function getYpValue(annotatedValue: AnnotatedValue) {
   const { $type } = annotatedValue;

   if (parseAnnotated.hasOwnProperty($type)) {
      return parseAnnotated[$type as KnownAnnotatedTypes](annotatedValue);
   }

   return annotatedValue;
}
