import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { Loader, Button } from '@yandex-cloud/uikit';
import { useSelector } from 'react-redux';
import { RouteComponentProps, useHistory } from 'react-router';

import { PageTemplate } from '../../../../components/yp';
import { YpLocation } from '../../../../models/api';
import { ReplicaSetFiltersParams } from '../../../../models/ui/yp/models';
import { getYPFilter, RootState } from '../../../../redux';
import { useYpReplicaSets } from '../../../../redux/hooks';
import { setUrlQuery } from '../../../../utils';
import { getReplicaSetQueryFromFilter } from '../../../../utils/yp';
import { getNavigation, parseFiltersValuesFromSearch } from '../../utils';
import { ReplicaSetFilters } from '../../../../components/yp/ReplicaSetFilters/ReplicaSetFilters';
import { selectYpReplicaSets } from '../../../../redux/slices';
import { ReplicaSetsTable } from '../../../../components/yp/ReplicaSetsTable/ReplicaSetsTable';
import { YpEntityRouteProps } from '../../../../models/ui/yp/view';

import classes from './ReplicaSets.module.css';

const REQUEST_KEY = 'ypReplicaSetsRequest';

const defaultFilters: ReplicaSetFiltersParams = {};

export const ReplicaSets: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({
   match: {
      params: { cluster },
   },
   location,
}) => {
   const navigation = useMemo(() => getNavigation(cluster, { id: 'replica-sets', title: 'replica sets' }), [cluster]);
   const { requestReplicaSets, isFetching } = useYpReplicaSets(cluster as YpLocation, REQUEST_KEY);

   const history = useHistory();

   const [filters, setFilters] = useState<ReplicaSetFiltersParams>(defaultFilters);
   const filterExpression = useMemo(() => getReplicaSetQueryFromFilter(filters), [filters]);
   const replicaSetsFilterSelector = useCallback(
      (state: RootState) => getYPFilter(state, cluster as YpLocation, 'replicaSets', filterExpression),
      [cluster, filterExpression],
   );
   const { ids: replicaSetIds, canLoadMore = false } = useSelector(replicaSetsFilterSelector) ?? {};
   const replicaSetsSelector = useCallback(
      (state: RootState) => selectYpReplicaSets(state, cluster as YpLocation, replicaSetIds),
      [cluster, replicaSetIds],
   );
   const replicaSets = useSelector(replicaSetsSelector);

   useEffect(() => {
      if (!location.search) {
         setUrlQuery(history, location, defaultFilters, false);
         setFilters(defaultFilters);
         requestReplicaSets(filters, true);
      } else {
         const query = parseFiltersValuesFromSearch(location.search) as ReplicaSetFiltersParams;
         setFilters(query);
         requestReplicaSets(query, true);
      }
   }, []); //eslint-disable-line

   const handleFilters = (newFilters: ReplicaSetFiltersParams) => {
      setUrlQuery(history, location, newFilters, false);
      setFilters(newFilters);
      requestReplicaSets(newFilters, true);
   };

   const onLoadMore = useCallback(() => {
      requestReplicaSets(filters, false);
   }, [filters, requestReplicaSets]);

   return (
      <PageTemplate error={undefined} navigation={navigation}>
         <div>
            <ReplicaSetFilters onSubmit={handleFilters} filters={filters} disabled={false} />
         </div>
         {replicaSets && (
            <div>
               <ReplicaSetsTable replicaSets={replicaSets} cluster={cluster} type={'TReplicaSet'} />
            </div>
         )}
         <div className={classes.loaderContainer}>
            {isFetching && <Loader size={'s'} />}
            {canLoadMore && !isFetching && (
               <Button onClick={onLoadMore} view={'outlined'}>
                  Load more
               </Button>
            )}
         </div>
      </PageTemplate>
   );
};
