import React, { useCallback, useEffect, useMemo } from 'react';

import { Json } from '@yandex-infracloud-ui/libs';
import { RouteComponentProps } from 'react-router';
import { Loader } from '@yandex-cloud/uikit';

import { PageTemplate, Tab, Tabs } from '../../../../../components/yp';
import { YpLocation } from '../../../../../models/api';
import { useNetworkErrors, useYpPods } from '../../../../../redux';
import { useYpReplicaSet } from '../../../../../redux/hooks/useYpReplicaSet';
import { getNavigation, getYPError } from '../../../utils';
import { PodSetPods } from '../../pod-sets/_entityId/PodSetPods/PodSetPods';
import { ReplicaSetInfo } from '../../../../../components/yp/ReplicaSetInfo/ReplicaSetInfo';
import { PodsFiltersParams, YpEntityRouteProps } from '../../../../../models/ui/yp/view';

import classes from './ReplicaSet.module.css';

const networkRequests = {
   replicaSet: 'ypClusterReplicaSetRequest',
   pods: 'ypClusterReplicaSetPodsRequest',
};

export const ReplicaSet: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({
   match: {
      params: { cluster, entityId },
   },
   location,
}) => {
   const navigation = useMemo(() => getNavigation(cluster, 'replica-sets', entityId), [cluster, entityId]);
   const { replicaSet } = useYpReplicaSet(entityId, cluster as YpLocation, networkRequests.replicaSet);
   const podsFilter = useMemo<PodsFiltersParams>(
      () => ({
         podSetId: entityId,
      }),
      [entityId],
   );

   const { meta, labels, spec, status } = replicaSet ?? {};

   const { canFetch, isFetching, pods, requestPods } = useYpPods(
      cluster as YpLocation,
      networkRequests.pods,
      podsFilter,
   );

   useEffect(() => {
      requestPods(podsFilter, 1, true);
   }, []); //eslint-disable-line

   const errors = useNetworkErrors(Object.values(networkRequests));

   const replicaSetError = useMemo(() => getYPError(errors[networkRequests.replicaSet]), [errors]);

   const podsError = useMemo(() => getYPError(errors[networkRequests.pods]), [errors]);

   const onLoadMore = useCallback(
      newPage => {
         requestPods(
            {
               podSetId: entityId,
            },
            newPage,
            false,
         );
      },
      [requestPods, entityId],
   );

   if (!replicaSet && !Object.values(errors).length) {
      return (
         <div className={classes.loaderContainer}>
            <span className={classes.loader}>
               <Loader size={'s'} />
            </span>
            <span className={classes.loaderGap} />
         </div>
      );
   }

   const showPods = pods || podsError;

   const tabs: Tab[] = ([
      replicaSet && {
         id: 'info',
         title: 'General',
         content: (
            <div>
               <ReplicaSetInfo replicaSet={replicaSet} cluster={cluster} type={'TReplicaSet'} />
            </div>
         ),
      },
      meta && {
         id: 'meta',
         title: 'Meta',
         content: <Json obj={meta} />,
      },
      labels && {
         id: 'labels',
         title: 'Labels',
         content: <Json obj={labels} />,
      },
      spec && {
         id: 'spec',
         title: 'Spec',
         content: <Json obj={spec} />,
      },
      status && {
         id: 'status',
         title: 'Status',
         content: <Json obj={status} />,
      },
   ] as (Tab | undefined)[]).filter((item): item is Tab => Boolean(item));

   if (showPods) {
      tabs.push({
         id: 'pods',
         title: (
            <span>
               Pods{' '}
               {pods && (
                  <span className={classes.podsCount}>
                     {status?.deploy_status?.details?.total_progress?.pods_total ?? 0}
                  </span>
               )}
            </span>
         ),
         content: (
            <div>
               {podsError ? (
                  <div className={classes.podsError}>{podsError}</div>
               ) : (
                  <PodSetPods
                     cluster={cluster as YpLocation}
                     pods={pods}
                     canFetch={canFetch}
                     isFetching={isFetching}
                     onLoadMore={onLoadMore}
                  />
               )}
            </div>
         ),
         width: 'full',
      });
   }

   return (
      <PageTemplate error={replicaSetError} navigation={navigation}>
         <Tabs tabs={tabs} />
      </PageTemplate>
   );
};
