/* eslint-disable */

/**
 * EPodAgentTargetState_UNKNOWN -  Дефолтное значение
 *  Интерпретируется как ACTIVE
 *
 * EPodAgentTargetState_ACTIVE -  При этом target state все объекты, указанные в спецификации пода, будут собираться/пересобираться
 *  пока все не перейдут в ready state
 *
 * EPodAgentTargetState_SUSPENDED -  При этом target state все объекты кроме workloads, указанные в спецификации пода, будут собираться/пересобираться
 *  пока все не перейдут в ready state
 *  Если в момент перехода в SUSPENDED state были активные workloads, то у них будет вызвана stop_policy
 *  Под будет считаться готовым в тот момент, когда все workloads с указанным readiness check
 *  будут остановлены при помощи stop_policy
 *  При этом, если на момент остановки start процесс workload будет отсутствовать, то stop_policy вызвана не будет
 *
 * EPodAgentTargetState_REMOVED -  При этом target state все объекты внутри пода будут удалены
 *  Под будет считаться готовым в тот момент, когда все stop/destroy_policy у workloads с указанным readiness check
 *  будут выполнены
 *  При этом, если на момент остановки start процесс workload будет отсутствовать, то будет вызвана только destroy_policy,
 *  а вызов stop_policy будет пропущен
 *
 */
export enum EPodAgentTargetState {
   EPodAgentTargetState_UNKNOWN = 'unknown',
   EPodAgentTargetState_ACTIVE = 'active',
   EPodAgentTargetState_SUSPENDED = 'suspended',
   EPodAgentTargetState_REMOVED = 'removed',
}

/**
 *  "rt" cpu policy is forbidden: DEPLOY-2046
 * ECpuPolicy_UNKNOWN -  Use porto default cpu policy
 *
 * ECpuPolicy_NORMAL -  SCHED_OTHER (default)
 *
 * ECpuPolicy_HIGH -  SCHED_OTHER (nice = -10, increases cpu.shares by 16 times)
 *
 * ECpuPolicy_BATCH -  SCHED_BATCH
 *
 * ECpuPolicy_IDLE -  SCHED_IDLE (also decreases cpu.shares by 16 times)
 *
 * ECpuPolicy_ISO -  SCHED_ISO (offstream kernel feature)
 *
 */
export enum ECpuPolicy {
   ECpuPolicy_UNKNOWN = 'unknown',
   ECpuPolicy_NORMAL = 'normal',
   ECpuPolicy_HIGH = 'high',
   ECpuPolicy_BATCH = 'batch',
   ECpuPolicy_IDLE = 'idle',
   ECpuPolicy_ISO = 'iso',
}

/**
 *  "rt" io policy is forbidden: DEPLOY-2285
 * EIoPolicy_UNKNOWN -  Use porto default io policy
 *
 * EIoPolicy_NONE -  set by cpu_policy, blkio.weight = 500
 *
 * EIoPolicy_NORMAL -  IOPRIO_CLASS_BE(4), blkio.weight = 500
 *
 * EIoPolicy_HIGH -  IOPRIO_CLASS_BE(0), blkio.weight = 1000
 *
 * EIoPolicy_BATCH -  IOPRIO_CLASS_BE(7), blkio.weight = 10
 *
 * EIoPolicy_IDLE -  IOPRIO_CLASS_IDLE, blkio.weight = 10
 *
 */
export enum EIoPolicy {
   EIoPolicy_UNKNOWN = 'unknown',
   EIoPolicy_NONE = 'none',
   EIoPolicy_NORMAL = 'normal',
   EIoPolicy_HIGH = 'high',
   EIoPolicy_BATCH = 'batch',
   EIoPolicy_IDLE = 'idle',
}

export enum EIoLimitType {
   EIoLimitType_READ = 'read',
   EIoLimitType_WRITE = 'write',
}

/**
 * EDefaultIoLimitTarget_PLACE -  Соответствует "/place"
 *
 * EDefaultIoLimitTarget_SSD -  Соответствует "/ssd"
 *
 */
export enum EDefaultIoLimitTarget {
   EDefaultIoLimitTarget_NONE = 'none',
   EDefaultIoLimitTarget_PLACE = 'place',
   EDefaultIoLimitTarget_SSD = 'ssd',
}

/**
 * ETransmitSystemLogsPolicy_NONE -  Дефолтное значение
 *  Интерпретируется как DISABLED
 *
 * ETransmitSystemLogsPolicy_DISABLED -  Отсылка системных логов отключена
 *
 * ETransmitSystemLogsPolicy_ENABLED -  Отсылка системных логов включена
 *
 */
export enum ETransmitSystemLogs {
   ETransmitSystemLogsPolicy_NONE = 'none',
   ETransmitSystemLogsPolicy_DISABLED = 'disabled',
   ETransmitSystemLogsPolicy_ENABLED = 'enabled',
}

/**
 * EAutoDecodeBase64Secrets_NONE -  Дефолтное значение
 *  Интерпретируется как DISABLED
 *
 * EAutoDecodeBase64Secrets_DISABLED -  Декодировка base64 секретов отключена
 *
 * EAutoDecodeBase64Secrets_ENABLED -  Декодировка base64 секретов включена
 *
 */
export enum EAutoDecodeBase64Secrets {
   EAutoDecodeBase64Secrets_NONE = 'none',
   EAutoDecodeBase64Secrets_DISABLED = 'disabled',
   EAutoDecodeBase64Secrets_ENABLED = 'enabled',
}

/**
 * EMultiSecretFileContentFormat_JAVA -  В отличии от ya vault текст не перекодируется в latin1 и остается в исходной кодировке
 *
 */
export enum EMultiSecretFileContentFormat {
   EMultiSecretFileContentFormat_JSON = 'json',
   EMultiSecretFileContentFormat_JAVA = 'java',
   EMultiSecretFileContentFormat_YAML = 'yaml',
}

/**
 * ESkyGetDeduplicateMode_NO -  Режим дедупликации выключен
 *
 * ESkyGetDeduplicateMode_HARDLINK -  Формировать жесткую ссылку
 *
 * ESkyGetDeduplicateMode_SYMLINK -  Формировать символьную ссылку
 *
 */
export enum ESkyGetDeduplicateMode {
   ESkyGetDeduplicateMode_NO = 'no',
   ESkyGetDeduplicateMode_HARDLINK = 'hardlink',
   ESkyGetDeduplicateMode_SYMLINK = 'symlink',
}

export enum EVolumeMountMode {
   EVolumeMountMode_READ_ONLY = 'read_only',
   EVolumeMountMode_READ_WRITE = 'read_write',
}

export enum EVolumeCreateMode {
   EVolumeCreateMode_READ_WRITE = 'read_write',
   EVolumeCreateMode_READ_ONLY = 'read_only',
}

/**
 *  Deprecated
 */
export enum EResourceAccessMode {
   EResourceAccessMode_664 = '664',
   EResourceAccessMode_660 = '660',
   EResourceAccessMode_600 = '600',
}

export enum EResourceAccessPermissions {
   EResourceAccessPermissions_UNMODIFIED = 'unmodified',
   EResourceAccessPermissions_660 = '660',
   EResourceAccessPermissions_600 = '600',
}

/**
 * EContainerIsolationMode_CHILD_ONLY - Из контейнера и его подконтейнеров нету доступа к соседним контейнерам
 *
 */
export enum EContainerIsolationMode {
   EContainerIsolationMode_NO_ISOLATION = 'no_isolation',
   EContainerIsolationMode_CHILD_ONLY = 'child_only',
}

export enum EVolumePersistenceType {
   EVolumePersistenceType_PERSISTENT = 'persistent',
   EVolumePersistenceType_NON_PERSISTENT = 'non_persistent',
}

/**
 * EContainerULimit_AS -  address space limit
 *
 * EContainerULimit_CORE -  limits the core file size
 *
 * EContainerULimit_DATA -  max data size
 *
 * EContainerULimit_MEMLOCK -  max locked-in-memory address space
 *
 * EContainerULimit_STACK -  max stack size
 *
 * EContainerULimit_FSIZE -  maximum filesize
 *
 * EContainerULimit_NOFILE -  max number of open files
 *
 * EContainerULimit_CPU -  max CPU time
 *
 * EContainerULimit_LOCKS -  max number of file locks the user can hold
 *
 * EContainerULimit_MSGQUEUE -  max memory used by POSIX message queues
 *
 * EContainerULimit_NICE -  max nice priority allowed to raise to values
 *
 * EContainerULimit_NPROC -  max number of processes
 *
 * EContainerULimit_RSS -  max resident set size
 *
 * EContainerULimit_RTPRIO -  max realtime priority
 *
 * EContainerULimit_RTTIME -  timeout for real-time tasks
 *
 * EContainerULimit_SIGPENDING -  max number of pending signals
 *
 */
export enum EContainerULimitType {
   EContainerULimit_AS = 'as',
   EContainerULimit_CORE = 'core',
   EContainerULimit_DATA = 'data',
   EContainerULimit_MEMLOCK = 'memlock',
   EContainerULimit_STACK = 'stack',
   EContainerULimit_FSIZE = 'fsize',
   EContainerULimit_NOFILE = 'nofile',
   EContainerULimit_CPU = 'cpu',
   EContainerULimit_LOCKS = 'locks',
   EContainerULimit_MSGQUEUE = 'msgqueue',
   EContainerULimit_NICE = 'nice',
   EContainerULimit_NPROC = 'nproc',
   EContainerULimit_RSS = 'rss',
   EContainerULimit_RTPRIO = 'rtprio',
   EContainerULimit_RTTIME = 'rttime',
   EContainerULimit_SIGPENDING = 'sigpending',
}

/**
 * EResolvConf_DEFAULT -  Использовать конфигурацию из portod.conf с хоста
 *  Если в portod.conf нет конфигурации, будет использован resolv.conf с хоста
 *
 * EResolvConf_KEEP -  Оставляет resolv.conf в контейнере as is, никак не изменяя его
 *
 * EResolvConf_NAT64 -  При использовании nat64 resolv.conf будет содержать следующие строки:
 *  1) search search.yandex.net.
 *  2) nameserver 2a02:6b8:0:3400::5005 (https://wiki.yandex-team.ru/noc/ipv6/statefulnat64)
 *  3) options timeout:1 attempts:1
 *  Возможно, список изменится https://st.yandex-team.ru/DEPLOY-1436#5ddcf980701665001d5c3459
 *
 * EResolvConf_NAT64_LOCAL -  При использовании nat64_local resolv.conf будет содержать следующие строки:
 *  1) search search.yandex.net.
 *  2) nameserver fd64::1 (https://st.yandex-team.ru/SPPROBLEM-202#5e26c612e5106644f8457d51)
 *  3) nameserver 2a02:6b8:0:3400::5005 (https://wiki.yandex-team.ru/noc/ipv6/statefulnat64)
 *  4) options timeout:1 attempts:1
 *
 */
export enum EResolvConf {
   EResolvConf_DEFAULT = 'default',
   EResolvConf_KEEP = 'keep',
   EResolvConf_NAT64 = 'nat64',
   EResolvConf_NAT64_LOCAL = 'nat64_local',
}

export enum EWorkloadTargetState {
   EWorkloadTarget_ACTIVE = 'active',
   EWorkloadTarget_REMOVED = 'removed',
}

/**
 * EUnixSignalType_DEFAULT -  Интерпретируется как SIGTERM
 *
 */
export enum EUnixSignalType {
   EUnixSignalType_DEFAULT = 'DEFAULT',
   EUnixSignalType_SIGHUP = 'SIGHUP',
   EUnixSignalType_SIGINT = 'SIGINT',
   EUnixSignalType_SIGALRM = 'SIGALRM',
   EUnixSignalType_SIGKILL = 'SIGKILL',
   EUnixSignalType_SIGTERM = 'SIGTERM',
   EUnixSignalType_SIGUSR1 = 'SIGUSR1',
   EUnixSignalType_SIGUSR2 = 'SIGUSR2',
   EUnixSignalType_SIGURG = 'SIGURG',
   EUnixSignalType_SIGQUIT = 'SIGQUIT',
}

/**
 *  Определяет, нужно ли монтирование fs cgroup в бокс и с какими правами
 */
export enum ECgroupFsMountMode {
   ECgroupFsMountMode_NONE = 'none',
   ECgroupFsMountMode_RO = 'ro',
   ECgroupFsMountMode_RW = 'rw',
}

/**
 * ELayerSourceFileStoragePolicy_NONE -  Дефолтное значение для сохранения обратной совместимости
 *  Эквивалентно ELayerSourceFileStoragePolicy_REMOVE
 *
 * ELayerSourceFileStoragePolicy_KEEP -  Сохранять исходные файлы лейров
 *
 * ELayerSourceFileStoragePolicy_REMOVE -  Удалять исходные файлы лейров
 *
 */
export enum ELayerSourceFileStoragePolicy {
   ELayerSourceFileStoragePolicy_NONE = 'none',
   ELayerSourceFileStoragePolicy_KEEP = 'keep',
   ELayerSourceFileStoragePolicy_REMOVE = 'remove',
}

/**
 * EBoxState_WAITING_FOR_ROOTFS_LAYERS -  Ждет пока заимпортится rootfs layers в porto
 *
 * EBoxState_CREATING_ROOTFS_VOLUME -  В процессе сборки rootfs volume
 *
 * EBoxState_WAITING_FOR_VOLUMES -  Ждет пока соберутся MountedVolumes
 *
 * EBoxState_LINKING_VOLUMES -  Приликовывает MountedVolumes к RootfsVolume
 *
 * EBoxState_INIT_PROCESSES -  Запуск Init процессов
 *
 * EBoxState_READY -  Вольюмы смонтированы, окружение готово к запуску хуков
 *
 * EBoxState_REMOVING -  Соотвествует процессу удаления box
 *
 * EBoxState_REMOVED -  Соотвестует удалённому box
 *
 * EBoxState_INVALID -  Box находится в невалидном состоянии и значит будет полнотью пересобран
 *
 * EBoxState_WAITING_FOR_STATIC_RESOURCES -  Ждет пока заимпортятся статические ресурсы
 *
 * EBoxState_CREATING_ROOT_CONTAINER -  В процессе сборки root container
 *
 */
export enum EBoxState {
   EBoxState_UNKNOWN = 'unknown',
   EBoxState_WAITING_FOR_ROOTFS_LAYERS = 'waiting_for_rootfs_layers',
   EBoxState_CREATING_ROOTFS_VOLUME = 'creating_rootfs_volume',
   EBoxState_WAITING_FOR_VOLUMES = 'waiting_for_volumes',
   EBoxState_LINKING_VOLUMES = 'linking_volumes',
   EBoxState_INIT_PROCESSES = 'init_processes',
   EBoxState_READY = 'ready',
   EBoxState_REMOVING = 'removing',
   EBoxState_REMOVED = 'removed',
   EBoxState_INVALID = 'invalid',
   EBoxState_WAITING_FOR_STATIC_RESOURCES = 'waiting_for_static_resources',
   EBoxState_CREATING_ROOT_CONTAINER = 'creating_root_container',
}

/**
 * EWorkloadState_WAITING_FOR_BOX -  Ожидание готовности box
 *
 * EWorkloadState_ACTIVATING -  Процесс start активен и readiness еще ни разу не завершился
 *
 * EWorkloadState_ACTIVE -  Процесс start активен и последний запуск readiness завершился успешно
 *
 * EWorkloadState_DEACTIVATING -  Остановка процесса start - работа stop policy
 *
 * EWorkloadState_REMOVING -  Удаление workload - работа destroy policy
 *
 * EWorkloadState_REMOVED -  Workload удален
 *
 * EWorkloadState_INVALID -  Workload находится в невалидном состоянии и будет полностью пересобран
 *
 * EWorkloadState_DEACTIVATED_BY_LIVENESS -  Workload был убит из-за неудачного завершения liveness
 *
 * EWorkloadState_INIT_PROCESSES -  Выполняются init команды
 *
 * EWorkloadState_DEAD -  Процесс start завершился (сам, либо был убит) и будет перезапущен
 *
 * EWorkloadState_READINESS_FAILED -  Процесс start активен и последний запуск readiness завершился ошибкой
 *
 * EWorkloadState_FAILED_TO_START_READINESS -  При попытке сделать запуск readiness произошла системная ошибка
 *
 * EWorkloadState_FAILED_TO_START_LIVENESS -  При попытке сделать запуск liveness произошла системная ошибка
 *
 */
export enum EWorkloadState {
   EWorkloadState_UNKNOWN = 'unknown',
   EWorkloadState_WAITING_FOR_BOX = 'waiting_for_box',
   EWorkloadState_ACTIVATING = 'activating',
   EWorkloadState_ACTIVE = 'active',
   EWorkloadState_DEACTIVATING = 'deactivating',
   EWorkloadState_REMOVING = 'removing',
   EWorkloadState_REMOVED = 'removed',
   EWorkloadState_INVALID = 'invalid',
   EWorkloadState_DEACTIVATED_BY_LIVENESS = 'deactivated_by_liveness',
   EWorkloadState_INIT_PROCESSES = 'init_processes',
   EWorkloadState_DEAD = 'dead',
   EWorkloadState_READINESS_FAILED = 'readiness_failed',
   EWorkloadState_FAILED_TO_START_READINESS = 'failed_to_start_readiness',
   EWorkloadState_FAILED_TO_START_LIVENESS = 'failed_to_start_liveness',
}

/**
 *  Описание результата выполнения хука
 * EContainerState_EXITED -  Процесс завершился.
 *
 * EContainerState_OUT_OF_MEMORY -  Процесс убили из-за превышения памяти
 *
 * EContainerState_TIMEOUT -  Произошел таймаут выполенния
 *
 * EContainerState_KILLED_EXTERNALLY -  Процесс был убит извне (кто-то вызвал kill или destroy)
 *
 * EContainerState_SYSTEM_FAILURE -  Произошла системная ошибка (например ошибка от порто или т.п.)
 *
 * EContainerState_RUNNING -  Процесс запущен и работает
 *
 * EContainerState_WAITING_INIT -  Процесс ожидает истечения InitialDelayMs
 *
 * EContainerState_WAITING_RESTART -  Процесс ожидает истечения restart period - @see TTimeLimit
 *
 */
export enum EContainerState {
   EContainerState_UNKNOWN = 'unknown',
   EContainerState_EXITED = 'exited',
   EContainerState_OUT_OF_MEMORY = 'out_of_memory',
   EContainerState_TIMEOUT = 'timeout',
   EContainerState_KILLED_EXTERNALLY = 'killed_externally',
   EContainerState_SYSTEM_FAILURE = 'system_failure',
   EContainerState_RUNNING = 'running',
   EContainerState_WAITING_INIT = 'waiting_init',
   EContainerState_WAITING_RESTART = 'waiting_restart',
}

export enum EConditionStatus {
   EConditionStatus_UNKNOWN = 'unknown',
   EConditionStatus_TRUE = 'true',
   EConditionStatus_FALSE = 'false',
}

/**
 * EHttpGetState_FAILURE -  Ошибка во время выполнения (превышен таймаут, ошибка соединения, ...)
 *
 * EHttpGetState_WRONG_ANSWER -  Ответ не равен THttpGet::expected_answer, только для Readiness check и Liveness check
 *
 * EHttpGetState_RUNNING -  Выполняется
 *
 * EHttpGetState_WAITING_INIT -  Ожидание истечения InitialDelayMs
 *
 * EHttpGetState_WAITING_RESTART -  Ожидание истечения restart period - @see TTimeLimit
 *
 * EHttpGetState_SUCCESS -  Http запрос выполнился с 200-299 кодом возврата и ответ соответствует THttpGet::expected_answer
 *
 */
export enum EHttpGetState {
   EHttpGetState_UNKNOWN = 'unknown',
   EHttpGetState_FAILURE = 'failure',
   EHttpGetState_WRONG_ANSWER = 'wrong_answer',
   EHttpGetState_RUNNING = 'running',
   EHttpGetState_WAITING_INIT = 'waiting_init',
   EHttpGetState_WAITING_RESTART = 'waiting_restart',
   EHttpGetState_SUCCESS = 'success',
}

/**
 * ETcpCheckState_FAILURE -  Connect не завершился успешно
 *
 * ETcpCheckState_RUNNING -  Выполняется
 *
 * ETcpCheckState_WAITING_INIT -  Ожидание истечения InitialDelayMs
 *
 * ETcpCheckState_WAITING_RESTART -  Ожидание истечения restart period - см. TTimeLimit
 *
 * ETcpCheckState_SUCCESS -  Connect завершился успешно
 *
 */
export enum ETcpCheckState {
   ETcpCheckState_UNKNOWN = 'unknown',
   ETcpCheckState_FAILURE = 'failure',
   ETcpCheckState_RUNNING = 'running',
   ETcpCheckState_WAITING_INIT = 'waiting_init',
   ETcpCheckState_WAITING_RESTART = 'waiting_restart',
   ETcpCheckState_SUCCESS = 'success',
}

/**
 * EUnixSignalState_FAILURE -  Произошла ошибка при попытке послать сигнал
 *
 * EUnixSignalState_SUCCESS -  Сигнал был послан успешно
 *
 * EUnixSignalState_WAITING_INIT -  Ожидание истечения InitialDelayMs
 *
 * EUnixSignalState_WAITING_RESTART -  Ожидание истечения restart period - см. TTimeLimit
 *
 */
export enum EUnixSignalState {
   EUnixSignalState_UNKNOWN = 'unknown',
   EUnixSignalState_FAILURE = 'failure',
   EUnixSignalState_SUCCESS = 'success',
   EUnixSignalState_WAITING_INIT = 'waiting_init',
   EUnixSignalState_WAITING_RESTART = 'waiting_restart',
}

/**
 *  Ресурс при скачивании и проверке проиходит через несколько стадий
 * ELayerState_DOWNLOADING -  В процессе скачивания
 *
 * ELayerState_VERIFYING -  Проверка контрольной суммы
 *
 * ELayerState_IMPORTING -  Импортируется в porto
 *
 * ELayerState_READY -  Готов к использованию
 *
 * ELayerState_REMOVING -  Соответствует процессу удаления ресурса
 *
 * ELayerState_REMOVED -  Соответствует удалённому ресурсу
 *
 * ELayerState_INVALID -  Layer находится в невалидном состоянии и значит будет полнотью пересобран
 *
 * ELayerState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER -  Соответствует процессу создания meta контейнера для загрузки ресурсов
 *
 * ELayerState_IN_QUEUE -  Ожидания момента, когда одновременно будет работать меньше TResourceGang.active_download_containers_limit контейнеров для загрузки ресурсов
 *
 * ELayerState_VERIFICATION_IN_QUEUE -  Ожидания момента, когда одновременно будет работать меньше TResourceGang.active_verify_containers_limit контейнеров для верификации ресурсов
 *
 */
export enum ELayerState {
   ELayerState_UNKNOWN = 'unknown',
   ELayerState_DOWNLOADING = 'downloading',
   ELayerState_VERIFYING = 'verifying',
   ELayerState_IMPORTING = 'importing',
   ELayerState_READY = 'ready',
   ELayerState_REMOVING = 'removing',
   ELayerState_REMOVED = 'removed',
   ELayerState_INVALID = 'invalid',
   ELayerState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER = 'waiting_for_resource_gang_meta_container',
   ELayerState_IN_QUEUE = 'in_queue',
   ELayerState_VERIFICATION_IN_QUEUE = 'verification_in_queue',
}

/**
 *  Статический ресурс при скачивании и проверке проиходит через несколько стадий
 * EStaticResourceState_DOWNLOADING -  В процессе скачивания
 *
 * EStaticResourceState_VERIFYING -  Проверка контрольной суммы
 *
 * EStaticResourceState_REMOVING -  Соответствует процессу удаления ресурса
 *
 * EStaticResourceState_REMOVED -  Соответствует удалённому ресурсу
 *
 * EStaticResourceState_READY -  Готов к использованию
 *
 * EStaticResourceState_INVALID -  Ресурс находится в невалидном состоянии
 *
 * EStaticResourceState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER -  Соответствует процессу создания meta контейнера для загрузки ресурсов
 *
 * EStaticResourceState_IN_QUEUE -  Ожидания момента, когда одновременно будет работать меньше TResourceGang.active_download_containers_limit контейнеров для загрузки ресурсов
 *
 * EStaticResourceState_VERIFICATION_IN_QUEUE -  Ожидания момента, когда одновременно будет работать меньше TResourceGang.active_verify_containers_limit контейнеров для верификации ресурсов
 *
 */
export enum EStaticResourceState {
   EStaticResourceState_UNKNOWN = 'unknown',
   EStaticResourceState_DOWNLOADING = 'downloading',
   EStaticResourceState_VERIFYING = 'verifying',
   EStaticResourceState_REMOVING = 'removing',
   EStaticResourceState_REMOVED = 'removed',
   EStaticResourceState_READY = 'ready',
   EStaticResourceState_INVALID = 'invalid',
   EStaticResourceState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER = 'waiting_for_resource_gang_meta_container',
   EStaticResourceState_IN_QUEUE = 'in_queue',
   EStaticResourceState_VERIFICATION_IN_QUEUE = 'verification_in_queue',
}

/**
 * EVolumeState_WAITING_FOR_LAYERS -  Ждет пока заимпортятся layers в porto
 *
 * EVolumeState_CREATING -  В процессе сборки
 *
 * EVolumeState_READY -  Готов
 *
 * EVolumeState_REMOVING -  Соотвествует процессу удаления volume
 *
 * EVolumeState_REMOVED -  Соотвестует удалённому volume
 *
 * EVolumeState_INVALID -  Volume находится в не валидном состоянии и значит будет полнотью пересобран
 *
 * EVolumeState_WAITING_FOR_STATIC_RESOURCES -  Ждет пока подготовятся статические ресурсы
 *
 */
export enum EVolumeState {
   EVolumeState_UNKNOWN = 'unknown',
   EVolumeState_WAITING_FOR_LAYERS = 'waiting_for_layers',
   EVolumeState_CREATING = 'creating',
   EVolumeState_READY = 'ready',
   EVolumeState_REMOVING = 'removing',
   EVolumeState_REMOVED = 'removed',
   EVolumeState_INVALID = 'invalid',
   EVolumeState_WAITING_FOR_STATIC_RESOURCES = 'waiting_for_static_resources',
}

/**
 *  Конфигурация Пода, передаваемая в Pod-agent.
 *  Сущности верхнего уровня имеют уникальные в рамках Пода идентификаторы (поле Id)
 *  и могут ссылаться друг на друга по этим идентификаторам (поля *Refs).
 *  Валидация: нет висячих ссылок
 *  Валидация: на каждый Workload ссылается ровно один MutableWorkload, нет ссылок из MutableWorkload на несуществующие Workload
 */
export interface TPodAgentSpec {
   /**
    *  Pod revision, server-side монотонно возрастаюшая последовательность
    *  Стоит использовать только для связки объектов в пользовательских интерфейсах
    *  "Актуальность" объекта надо проверять по равенству spec_timestamp спеки/статуса
    */
   revision: number;
   /**
    *  Идентификатор пода
    */
   id: string;
   /**
    *  Слои и ресурсы, которые необходимо скачать в рамках лимитов @see ResourceGang
    */
   resources: TResourceGang | undefined;
   /**
    *  Все вольюмы
    */
   volumes: TVolume[];
   /**
    *  Все workload, изменение любого поля приводит к рестарту @see MutableWorkloads
    */
   workloads: TWorkload[];
   /**
    *  Все окружения, в которых могут быть запущены Workloads
    */
   boxes: TBox[];
   /**
    *  Часть Workload, изменение которой не приведет к полной пересборке Workload
    */
   mutable_workloads: TMutableWorkload[];
   /**
    *  Текущий target state для pod_agent
    *  В зависимости от него pod_agent будет собирать или разбирать объекты внутри пода
    */
   target_state: EPodAgentTargetState;
   /**
    *  Политика отсылки системных логов
    */
   transmit_system_logs_policy: TTransmitSystemLogsPolicy | undefined;
   /**
    *  Декодировка base64 секретов
    */
   auto_decode_base64_secrets: EAutoDecodeBase64Secrets;
}

/**
 *  Кэш ресурсы, которые надо скачать на под.
 *  Хеш ресурса считается как hash(Url, Checksum, Place), т.е. он не зависит от Id/Revision/Meta/CheckPeriodMs/.. :
 *    если в спеке пода окажется layer или ресурс, у которого совпал хеш с каким-то ресурсом из кэша,
 *    то этот ресурс не будет закачиваться заново, а просто подставится из кэша.
 *  Для всех TResource с одинаковым хешом выбирается минимальное из времен TVerification::CheckPeriodMs :
 *    т.е. если в спеку пода придет TResource, который лежит в кэше, но его период верификации меньше
 *    периода верификации кэш ресурса, этот ресурс начнет верифицироваться с меньшим периодом
 *  Валидация: сущности имеют уникальные в рамках пода идентификаторы (Пара Id, Revision)
 */
export interface TPodAgentResourceCacheSpec {
   /**
    *  Кэш слои
    */
   layers: TCacheLayer[];
   /**
    *  Кэш ресурсы
    */
   static_resources: TCacheResource[];
   /**
    *  Дефолтная политика сохранения/удаления исходных файлов лейров
    */
   default_layer_source_file_storage_policy: ELayerSourceFileStoragePolicy;
}

/**
 *  Текущее состояние Пода, возвращаемое из Pod-agent
 */
export interface TPodAgentStatus {
   /**
    *  Последняя известная pod revision
    */
   revision: number;
   /**
    *  Идентификатор пода
    */
   id: string;
   /**
    *  Момент отправки сообщения с агента
    */
   host_timestamp: number;
   volumes: TVolumeStatus[];
   resource_gang: TResourceGangStatus | undefined;
   workloads: TWorkloadStatus[];
   boxes: TBoxStatus[];
   /**
    *  Если хотя бы один из Ready вложенных TWorkloadStatus имеет Status == EConditionStatus_UNKNOWN,
    *  обращается в EConditionStatus_UNKNOWN.
    *  Иначе, если хотя бы один из них имеет Status == EConditionStatus_FALSE,
    *  обращается в EConditionStatus_FALSE.
    *  Иначе обращается в EConditionStatus_TRUE.
    *  LastTransitionTime -- максимальное значение Ready.LastTransitionTime среди всех вложеных TWorkloadStatus
    */
   ready: TCondition | undefined;
   /**
    *  Если хотя бы один из InProgress вложенных T*Status имеет Status == EConditionStatus_UNKNOWN,
    *  обращается в EConditionStatus_UNKNOWN.
    *  Иначе, если хотя бы один из них имеет Status == EConditionStatus_TRUE,
    *  обращается в EConditionStatus_TRUE.
    *  Иначе обращается в EConditionStatus_FALSE.
    *  LastTransitionTime -- максимальное значение InProgress.LastTransitionTime среди всех вложеных T*Status
    */
   in_progress: TCondition | undefined;
   /**
    *  Если хотя бы один из Failed вложенных T*Status имеет Status == EConditionStatus_UNKNOWN,
    *  обращается в EConditionStatus_UNKNOWN.
    *  Иначе, если хотя бы один из них имеет Status == EConditionStatus_TRUE,
    *  обращается в EConditionStatus_TRUE.
    *  Иначе обращается в EConditionStatus_FALSE.
    *  LastTransitionTime -- максимальное значение InProgress.LastTransitionTime среди всех вложеных T*Status
    */
   failed: TCondition | undefined;
   resource_cache: TResourceGangStatus | undefined;
   /**
    *  yp timestamp последнего обновления спецификации пода
    */
   spec_timestamp: number;
   /**
    *  target state последнего обновления спецификации пода
    *  unknown интерпритируется как active
    */
   target_state: EPodAgentTargetState;
}

export interface TTransmitSystemLogsPolicy {
   transmit_system_logs: ETransmitSystemLogs;
}

export interface TIoLimit {
   /**
    *  Файловая система или путь до файла или путь до диска (fs|<path>|<disk>)
    *  Данная строчка будет передана в porto as is
    *  Чтобы указать девайс, укажите его название без префикса /dev/
    *  Чтобы указать путь относительно корня dom0 машины, укажите его с "/" в начале
    *  Чтобы указать путь относительно chroot контейнера, укажите его с "./" в начале
    */
   raw: string | undefined;
   /**
    *  Один из стандартных путей для io limit
    *  Например "/place" или "/ssd" на dom0 хосте
    */
   default_target: EDefaultIoLimitTarget | undefined;
   /**
    *  Virtual disk, на который в целом будет наложен лимит
    */
   virtual_disk_id_ref: string | undefined;
   /**
    *  Тип лимита (чтение или запись)
    */
   type: EIoLimitType;
   /**
    *  Величина лимита
    */
   value: number;
}

export interface TComputeResources {
   /**
    *  Ограничения по CPU, в виртуальных миллиядрах (vcpu).
    *  Тем самым, заказав vcpu = 1000, мы получим одно виртуальное ядро.
    */
   vcpu_guarantee: number;
   vcpu_limit: number;
   /**
    *  Параметры CPU scheduler
    */
   cpu_policy: ECpuPolicy;
   cpu_weight: number;
   /**
    *  Ограничения по памяти, в байтах
    */
   memory_guarantee: number;
   memory_limit: number;
   anonymous_memory_limit: number;
   /**
    *  Если true иммигрирует кэш при незначительной ошибке страницы, по умолчанию: false.
    */
   recharge_on_pgfault: boolean;
   thread_limit: number;
   /**
    *  Лимит на чтение/запись в байтах в секунду
    */
   io_limit: TIoLimit[];
   /**
    *  Лимит iops (input/output operations per second)
    */
   io_ops_limit: TIoLimit[];
   io_policy: EIoPolicy;
   io_weight: number;
}

/**
 *  Проверки целостности ресурса
 */
export interface TVerification {
   /**
    *  Контрольная сумма, формат <схема>:<значение>
    *  Поддерживаемые схемы "MD5:", "EMPTY:", "SHA256:" - case sensitive
    */
   checksum: string;
   /**
    *  Периодичность проверки контрольной суммы. Первая проверка выполняется сразу после завершения скачивания.
    *  Если проверка упадет, то все боксы (и ворклоды в них), в которые смонтирован ресурс, будут убиты c SIGKILL,
    *  ресурс будет перезакачен, после чего боксы будут пересобраны.
    *  Если равен 0, то будет интерпретироваться как 180000 (3 минуты)
    */
   check_period_ms: number;
   /**
    *  Если true, то ресурс не будет проверяться на checksum.
    */
   disabled: boolean;
}

export interface TSandboxResource {
   task_type: string;
   task_id: string;
   resource_type: string;
   resource_id: string;
   /**
    *  Attributes of sandbox resource, e.g.
    *    "backup_task": "577303390",
    *    "released": "testing",
    *    "ttl": "inf",
    *    "version": "1.200"
    */
   attributes: { [key: string]: string };
}

export interface TSandboxResource_AttributesEntry {
   key: string;
   value: string;
}

export interface TResourceMeta {
   sandbox_resource: TSandboxResource | undefined;
}

export interface TMultiSecretFileContent {
   /**
    *  Формат в котором будут записаны секреты
    *  Форматы аналогичны ya vault get version --format
    */
   format: EMultiSecretFileContentFormat;
   /**
    *  Alias секрета из которого будут выписаны все значения
    */
   secret_alias: string;
}

export interface TFile {
   /**
    *  Имя файла
    *  Валидация: не содержит "/"
    */
   file_name: string;
   /**
    *  Содержимое этой строки будет скопировано в файл
    */
   raw_data: string | undefined;
   /**
    *  Cодержимое секрета будет скопировано в файл
    */
   secret_data: SecretSelector | undefined;
   /**
    *  Файл со всеми полями из одного секрета
    *  Вывод похож на ya vault get version, только без дополнительной информации (Файл будет содержать только "value")
    */
   multi_secret_data: TMultiSecretFileContent | undefined;
}

export interface TFiles {
   /**
    *  Валидация: все имена файлов различны
    */
   files: TFile[];
}

/**
 *  Документация по параметрам https://doc.yandex-team.ru/Search/skynet-dg/concepts/sky-get.html#request-format
 */
export interface TSkyGetDownload {
   /**
    *  Идентификатор ресурса
    *  Например rbtorrent:<идентификатор ресурса>
    */
   resid: string;
   /**
    *  Применить дедупликацию при скачивании файлов (выполняется проверка на совпадение пользователя и контрольной суммы)
    */
   deduplicate: ESkyGetDeduplicateMode;
}

export interface TGroupId {
   value: number;
}

export interface TResource {
   /**
    *  Уникальное имя ресурса (в рамках pod'а).
    *  Должно состоять только из символов, разрешенных в имени porto контейнера. (a..z, A..Z, 0..9, _-@:.)
    */
   id: string;
   /**
    *  Не нужно для Pod-agent
    *  Мета информация о ресурсе, необходимо для интеграции с CI/CD
    */
   meta: TResourceMeta | undefined;
   /**
    *  @see TVerification
    */
   verification: TVerification | undefined;
   /**
    *  URL для скачивания, протоколы: http, https, rbtorrent
    *  Для записи строчки в файл URL должен быть равен "raw:<строка>",
    *  в этом случае будет создан файл с названием raw_file и содержимым <строка>
    *  Только для тестов: можно указать путь к файлу на хосте "local:<filepath>"
    */
   url: string | undefined;
   /**
    *  Указать набор файлов внутри ресурса
    *  Внутри каждого из файлов могут быть записаны raw данные или секреты
    */
   files: TFiles | undefined;
   /**
    *  Вызов sky get с особыми параметрами
    */
   sky_get: TSkyGetDownload | undefined;
   default_storage: boolean | undefined;
   /**
    *  Указатель на файловый сторедж
    */
   file_storage_ref: string | undefined;
   /**
    *  Virtual disk, где будет расположен ресурс
    *  Ресурс будт скачан на porto volume, созданный на этом диске
    *  Валидация: должен совпадать с virtual_disk_id_ref у volume, box, которые используют этот ресурс
    */
   virtual_disk_id_ref: string;
   /**
    *  Deprecated
    */
   access_mode: EResourceAccessMode;
   /**
    *  group_id, выставляемая на файлы с ресурсом.
    *  Так как ресурсы качаются в контексте пода, а не пользовательского бокса,
    *  то в момент закачки подовый агент ничего не знает про пользовательские uid/guid,
    *  именно по этому group_id в данном случае задается в виде числового значения.
    *  По дефолту group_id на файлы со статическими ресурсами задается как loadbase
    */
   group_id: TGroupId | undefined;
   access_permissions: EResourceAccessPermissions;
}

export interface TCacheResource {
   /**
    *  Ревизия кэш ресурса
    */
   revision: number;
   resource: TResource | undefined;
}

export interface TLayer {
   /**
    *  Уникальное имя слоя (в рамках pod'а).
    *  Должно состоять только из символов, разрешенных в имени porto контейнера. (a..z, A..Z, 0..9, _-@:.)
    */
   id: string;
   /**
    *  Не нужно для Pod-agent
    *  Мета информация о ресурсе, необходимо для интеграции с CI/CD
    */
   meta: TResourceMeta | undefined;
   /**
    *  Контрольная сумма, формат <схема>:<значение>
    *  Поддерживаемые схемы "MD5:", "EMPTY:", "SHA256:" - case sensitive
    */
   checksum: string;
   /**
    *  URL для скачивания, протоколы: http, https, rbtorrent
    *  Для записи строчки в файл URL должен быть равен "raw:<строка>"
    *  Только для тестов: можно указать путь к файлу на хосте "local:<filepath>"
    */
   url: string | undefined;
   /**
    *  Вызов sky get с особыми параметрами
    */
   sky_get: TSkyGetDownload | undefined;
   /**
    *  Virtual disk, где будет расположен layer
    *  Слой будт скачан на porto volume, созданный на этом диске и импортирован в porto
    *  Place, в который импортирован данный слой, лежит на этом же диске
    *  Валидация: должен совпадать с virtual_disk_id_ref у volume, box, которые используют этот слой
    */
   virtual_disk_id_ref: string;
   /**
    *  Политика сохранения/удаления исходных файлов лейра
    *  Данная политика переопределяет default_layer_source_file_storage_policy из TResourceGang в случае,
    *  если выбрано значение, отличное от ELayerSourceFileStoragePolicy_NONE
    */
   layer_source_file_storage_policy: ELayerSourceFileStoragePolicy;
}

export interface TCacheLayer {
   /**
    *  Ревизия кэш слоя
    */
   revision: number;
   layer: TLayer | undefined;
}

/**
 *  Позволяет монитровать один вольюм в разные mountPoint
 */
export interface TMountedVolume {
   /**
    *  Id volume, который требуется подмонтировать
    */
   volume_ref: string | undefined;
   /**
    *  Rbind volume ref
    *  Директория будет примонтирована из специальной директории rootfs pod'a: <special_dir>/<rbind_volume_ref>
    */
   rbind_volume_ref: string | undefined;
   /**
    *  Точка монтирования
    */
   mount_point: string;
   /**
    *  Мод монтирования (read_only/read_write)
    */
   mode: EVolumeMountMode;
}

export interface TMountedStaticResource {
   /**
    *  Id static_resource, который требуется подмонтировать
    */
   resource_ref: string;
   /**
    *  Точка монтирования
    *  Будет создана директория с этим путем, внутри директории будет лежать скаченный ресурс
    */
   mount_point: string;
}

export interface TRootfsVolume {
   /**
    *  Квота на размер
    *  Поле не обрабатывается - porto не умеет во вложенные квоты по диску
    */
   quota_bytes: number;
   /**
    *  Слои, из которых собирается volume
    *  Как и в porto, слои идут в порядке top-layer;...;bottom-layer
    *  Валидация: ids в списке должны быть уникальные
    */
   layer_refs: string[];
   /**
    *  Мод создания (read_write/read_only)
    *  По умолчанию read_write
    */
   create_mode: EVolumeCreateMode;
}

export interface TFileStorage {
   /**
    *  Квота на размер
    *  Поле не обрабатывается - porto не умеет во вложенные квоты по диску
    */
   quota_bytes: number;
}

export interface TGenericVolume {
   /**
    *  Квота на размер
    *  Поле не обрабатывается - porto не умеет во вложенные квоты по диску
    */
   quota_bytes: number;
   /**
    *  Слои, из которых собирается volume
    *  Как и в porto, слои идут в порядке top-layer;...;bottom-layer
    *  Валидация: ids в списке должны быть уникальные
    */
   layer_refs: string[];
}

export interface TVolumeMountedStaticResource {
   /**
    *  Id static_resource, который требуется подмонтировать
    */
   resource_ref: string;
   /**
    *  Точка монтирования
    *  Будет создана директория с этим путем (путь относительно корня volume), внутри директории будет лежать скаченный ресурс
    */
   volume_relative_mount_point: string;
}

export interface TVolume {
   /**
    *  Уникальное имя volume (в рамках pod'а).
    *  Должно состоять только из символов, разрешенных в имени porto контейнера. (a..z, A..Z, 0..9, _-@:.)
    */
   id: string;
   /**
    *  TODO: not implemented
    */
   storage: TFileStorage | undefined;
   generic: TGenericVolume | undefined;
   /**
    *  Virtual disk, где будет расположен volume
    *  Volume будет создан в place, находящемся на этом виртуальном диске
    *  Валидация: должен совпадать с virtual_disk_id_ref у слоев, из которых собирается volume
    */
   virtual_disk_id_ref: string;
   /**
    *  Валидация: все mount_point различны
    */
   static_resources: TVolumeMountedStaticResource[];
   persistence_type: EVolumePersistenceType;
}

/**
 *  Ограничения на время исполнения и интервал между запусками хука.
 *  Все времена в миллисекундах, однако времена, возвращаемые porto,
 *  имеют точность до секунды.
 *  Поэтому все значения будут округляться вверх до секунд.
 *  maxExecutionTimeMs - ограничение на время выполнения.
 *  Интервал между запусками рассчитывается по формуле
 *  min(maxRestartPeriodMs, minRestartPeriodMs + restartPeriodScaleMs * (restartPeriodBackoff ^ step))
 *  step считается с 0 и равен max(0, ConsecutiveFailuresCounter - 1) для workload container/http stop и всех типов workload destroy
 *  и max(0, ConsecutiveSuccessesCounter - 1) для всех остальных хуков: все типы workload readiness, все типы workload liveness,
 *  workload init, workload unix signal stop
 */
export interface TTimeLimit {
   /**
    *  Если равен 0, то будет интерпретироваться как 5000 (5 секунд) для всех типов workload readiness/liveness и workload unix signal stop
    *  Для остальных контейнеров остается без изменений
    */
   initial_delay_ms: number;
   restart_period_scale_ms: number;
   restart_period_back_off: number;
   /**
    *  Если равен 0, то будет интерпретироваться как 18446744073709551615 (UINT64_MAX)
    */
   max_restart_period_ms: number;
   /**
    *  Если равен 0, то будет интерпретироваться как 30000 (30 секунд)
    */
   min_restart_period_ms: number;
   /**
    *  Если равен 0, то будет интерпретироваться как 1800000 (30 минут)
    */
   max_execution_time_ms: number;
}

export interface TTimeLimitStatus {
   consecutive_successes_counter: number;
   consecutive_failures_counter: number;
}

/**
 *  Alias - имя (ключ в мапе) секрета в <pod>/spec/secrets (и TPodAgentRequest.Secrets)
 *  Id - ключ в yav секрете
 *  В момент обработки спеки будет добавлена переменная окружения с значением равным value yav секрета
 */
export interface SecretSelector {
   alias: string;
   id: string;
   /**
    *  Если этот флаг выставлен, то значение секрета будет base64 декодированно
    *  Этот флаг нужен поскольку секретница кодирует содержимое секретов-файлов в base64
    *  Более подробно в DEPLOY-1336
    *  Deprecated
    */
   decode_base64: boolean;
}

export interface LiteralEnvSelector {
   value: string;
}

export interface TEnvVarValue {
   secret_env: SecretSelector | undefined;
   literal_env: LiteralEnvSelector | undefined;
}

export interface TEnvVar {
   name: string;
   value: TEnvVarValue | undefined;
}

export interface TUlimitSoft {
   name: EContainerULimitType;
   value: number;
}

export interface TWorkload {
   /**
    *  Уникальное имя workload (в рамках pod'а).
    *  Должно состоять только из символов, разрешенных в имени porto контейнера. (a..z, A..Z, 0..9, _-@:.)
    */
   id: string;
   /**
    *  Указатель на Box, в котором workload будет запущен
    */
   box_ref: string;
   /**
    *  Переменные окружения - прокидываются во все контейнеры ворклода
    */
   env: TEnvVar[];
   /**
    *  Soft лимиты окружения - прокидываются во все контейнеры ворклода
    */
   ulimit_soft: TUlimitSoft[];
   /**
    *  Набор команд, запускаемых перед стартом основного процесса.
    *  Гарантируется, что каждая отработает at least once.
    *  Если команда падает - она перезапускается до победного
    */
   init: TUtilityContainer[];
   /**
    *  Описание основного процесса
    */
   start: TUtilityContainer | undefined;
   /**
    *  @see TReadinessCheck
    */
   readiness_check: TReadinessCheck | undefined;
   /**
    *  @see TLivenessCheck
    */
   liveness_check: TLivenessCheck | undefined;
   /**
    *  @see TStopPolicy
    */
   stop_policy: TStopPolicy | undefined;
   /**
    *  @see TDestroyPolicy
    */
   destroy_policy: TDestroyPolicy | undefined;
   /**
    *  Передавать логи stdout и stderr в YT
    *  Должно быть выставлено не более одного из: этот флаг и start.stdout_file/stderr_file/stdout_and_stderr_limit
    */
   transmit_logs: boolean;
}

/**
 *  Mutable поля Workload
 */
export interface TMutableWorkload {
   /**
    *  Ссылка на workload
    */
   workload_ref: string;
   target_state: EWorkloadTargetState;
}

/**
 *  Посылает сигнал процессу
 */
export interface TUnixSignal {
   signal: EUnixSignalType;
   /**
    *  @see TTimeLimit
    */
   time_limit: TTimeLimit | undefined;
}

/**
 *  При переводе TMutableWorkload.TargetState в состояние REMOVED или обновления ревизии workload или любой его зависимости
 *  будет вызван stop hook, который должен выполнить gracefully завершение контейнера start
 *  Если после завершения stop hook контейнер start находится в состоянии dead или stopped, то он будет уничтожен,
 *  иначе stop hook будет перезапускаться до тех пор, пока контейнер start не перейдет в одно из этих состояний
 *  Если stop hook был запущен max_tries раз, но так и не перевел start в состоянии dead или stopped, то start будет немедленно убит
 *  Если stop hook выполняется при помощи контейнера и завершился с НУЛЕВЫМ кодом возврата, но start не в состоянии dead или stopped,
 *  то start будет немедленно убит
 *  Если stop hook выполняется при помощи http запроса и завершился успешно, но start не в состоянии dead или stopped,
 *  то start будет немедленно убит
 *  Если stop hook выполняется при помощи unix сигнала, то никаких дополнительных действий с start контейнером в случае
 *  успешной/неуспешной попытки послать сигнал не будет
 */
export interface TStopPolicy {
   container: TUtilityContainer | undefined;
   http_get: THttpGet | undefined;
   unix_signal: TUnixSignal | undefined;
   /**
    *  Количестно попыток выполнить gracefully stop перед тем, как убить процесс
    *  Валидация: должно быть > 0
    */
   max_tries: number;
}

/**
 *  При удалении workload из спеки будет вызван destroy hook,
 *  который выпонится только после окончания выполнения stop hook и перехода start контейнера в состояние dead.
 *  Destroy hook будет будет перезапускаться до того момента, пока хоть раз не выполнится успешно,
 *  или не провалится max_tries раз
 */
export interface TDestroyPolicy {
   container: TUtilityContainer | undefined;
   http_get: THttpGet | undefined;
   /**
    *  Количестно попыток выполнить gracefully destroy перед тем, как полностью удалить start контейнер workload
    *  Валидация: должно быть > 0
    */
   max_tries: number;
}

/**
 *  Обеспечивает изоляцию по диску и pidns
 *  Сеть между боксами не изолирована - workload'ы разных боксов могут общаться друг с другом по localhost
 */
export interface TBox {
   /**
    *  Уникальное имя box (в рамках pod'а).
    *  Должно состоять только из символов, разрешенных в имени porto контейнера. (a..z, A..Z, 0..9, _-@:.)
    */
   id: string;
   rootfs: TRootfsVolume | undefined;
   /**
    *  Валидация: mount_point не пересекается со static_resource
    *  Валидация: все mount_point различны
    */
   volumes: TMountedVolume[];
   /**
    *  Процессы, запускаемые при подготовке Box после создания вольюмов
    */
   init: TUtilityContainer[];
   /**
    *  Валидация: mount_point не пересекается с volume
    *  Валидация: все mount_point различны
    */
   static_resources: TMountedStaticResource[];
   /**
    *  Если указано true
    *  pod agent создает bind volume из <pod_chroot>/place/berkanavt/supervisor в <box_chroot>/place/berkanavt/supervisor
    *  На файловой системе box'а обязательно наличие директории /place/berkanavt/supervisor и следующих симлинков (без них скачивание/раздача работать не будут совсем):
    *    /skynet => /Berkanavt/supervisor/base/active
    *    /usr/local/bin/sky => /skynet/tools/sky
    *    /Berkanavt => /place/berkanavt
    *  Эти директории и симлинки можно добавить с помощью небольшого слоя sbr:922226367
    *  PS Если вы используете расширенные базовые образы, нужные симлинки в них уже есть https://wiki.yandex-team.ru/runtime-cloud/virtualimages/#rasshirennyjjobraz
    */
   bind_skynet: boolean;
   /**
    *  Ограничения на meta контейнер Box
    */
   compute_resources: TComputeResources | undefined;
   /**
    *  Параметры сети в Box
    *  Передается при помощи porto property в meta контейнер box
    */
   resolv_conf: EResolvConf;
   /**
    *  Переменные окружения - прокидываются во все init контейнеры бокса, meta контейнер бокса и во все вложенные workload
    *  При этом, если переменная окружения указана и в box env, и в workload env, то будет выбрано значение из workload env
    */
   env: TEnvVar[];
   /**
    *  Virtual disk, где будет расположен box
    *  Rootfs volume будет создан в place, находящемся на этом виртуальном диске
    *  Валидация: должен совпадать с virtual_disk_id_ref у слоев и ресурсов, из которых собирается box
    */
   virtual_disk_id_ref: string;
   /**
    *  Тип бокса
    *  Должен иметь значение "default", "system" или одно из значений в списке TProjectSpec.user_specific_box_types
    */
   specific_type: string;
   /**
    *  По умолчанию без изоляции
    */
   isolation_mode: EContainerIsolationMode;
   cgroup_fs_mount_mode: ECgroupFsMountMode;
}

/**
 *  Успешными кодами ответов считаются те, что лежат в полуинтервале [200, 300)
 */
export interface THttpGet {
   port: number;
   /**
    *  Путь к ручке, например: "/ping"
    *  Если путь пустой, то будет интерпретироваться как "/" из-за особенностей http клиента
    */
   path: string;
   /**
    *  В случае несовпадения ответа http запрос считается неуспешным
    */
   expected_answer: string | undefined;
   /**
    *  Если выставлено в true, ответ сервера игнорируется (проверяется только код ответа)
    */
   any: boolean | undefined;
   /**
    *  @see TTimeLimit
    */
   time_limit: TTimeLimit | undefined;
}

/**
 *  Проверяет только, что connect по данному порту успешно завершился
 */
export interface TTcpCheck {
   port: number;
   /**
    *  @see TTimeLimit
    */
   time_limit: TTimeLimit | undefined;
}

/**
 *  Проверяет, что start контейнер жив
 */
export interface TStartContainerAliveCheck {}

/**
 *  В случае возврата 0 контейнером или успешном завершении http/tcp hook - workload переходит в состояние ACTIVE,
 *  в других случаях SEMI_FAILED
 *  Нет такого кода возврата status hook, при котором контейнер будет убит
 *  Это соответствет текущему поведению instaceCTL и qloudinit
 *       https://wiki.yandex-team.ru/jandekspoisk/sepe/instancectl/#ortho-tags
 *       https://docs.platform.yandex-team.ru/doc/component#status_hook
 *  Если пользователь хочет пробпросить кастомный стуктурированный status, то рекомендуемое место для этого
 *  stdout этого хука
 */
export interface TReadinessCheck {
   container: TUtilityContainer | undefined;
   tcp_check: TTcpCheck | undefined;
   http_get: THttpGet | undefined;
   start_container_alive_check: TStartContainerAliveCheck | undefined;
   /**
    *  Минимальное число последовательных падений проверки после успешного выполнения, чтобы считать пробу упавшей
    *  Значение 0 воспринимается (по умолчанию) как 1
    */
   failure_threshold: number;
   /**
    *  Минимальное число последовательных успешных проверок после падения, чтобы считать пробу успешной
    *  Значение 0 воспринимается (по умолчанию) как 1
    */
   success_threshold: number;
}

/**
 *  В случае ненулевого возврата или не успешном завершении http/tcp hook - workload убивается,
 *  и переходит в состояние DEACTIVATED_BY_LIVENESS
 */
export interface TLivenessCheck {
   container: TUtilityContainer | undefined;
   tcp_check: TTcpCheck | undefined;
   http_get: THttpGet | undefined;
   /**
    *  Минимальное число последовательных падений проверки после успешного выполнения, чтобы считать пробу упавшей
    *  Значение 0 воспринимается (по умолчанию) как 1
    */
   failure_threshold: number;
}

/**
 *  Наследует env от TWorkload, если принадлежит ему
 *  Наследует env от TBox, если является init коммандой этого box
 */
export interface TUtilityContainer {
   /**
    *  Атрибуты контейнера (по сути это сквозной проброс опций Porto)
    *  Тут важно разделять интерфейс предоствляемый пользователю и внутренний между YP<->agent
    */
   compute_resources: TComputeResources | undefined;
   /**
    *  Будет вызван exec команды, она не выполняется внутри оболочки, поэтому команды содержащие "|", ">" не будут работать
    *  Если ваш слой содержит bash, вы можете выполнить такие команды следующим образом: bash -c "echo hello world > file"
    */
   command_line: string;
   /**
    *  @see TTimeLimit
    */
   time_limit: TTimeLimit | undefined;
   /**
    *  Параметры обработки корок
    */
   core_command: string;
   /**
    *  Пользователь, владеющий процессами контейнера
    *  Если пользователь не указан, то будет выставлен root
    */
   user: string;
   /**
    *  Группа, владеющая процессами контейнера
    *  Если группа не указана, то будет выставлен root
    */
   group: string;
   /**
    *  Рабочая директория контейнера
    */
   cwd: string;
   /**
    *  Путь относительно корня бокса до файла для stdout
    *  Если файл не указан, то он будет создан внутри porto, а не внутри бокса
    */
   stdout_file: string;
   /**
    *  Путь относительно корня бокса до файла для stderr
    *  Если файл не указан, то он будет создан внутри porto, а не внутри бокса
    */
   stderr_file: string;
   /**
    *  Ограничение на размер stdout и stderr
    *  Если выставлен 0, то будет использовано стандартное значение porto, например, для 4.18.20 ограничение будет 8388608
    */
   stdout_and_stderr_limit: number;
}

/**
 *  Позволяет выставить ограничение сверху на все закачки. Важным моментом является, что сумма лимитов
 *  ресурсов может быть больше, чем даже лимиты Pod. Поэтому pod-agent должен будет запускать закачки так:
 *  Снача запускаются закачки для активных workload и параллельно запускаются только те, которые можно разместить
 *  в рамках лимитов box
 */
export interface TResourceGang {
   layers: TLayer[];
   static_resources: TResource[];
   /**
    *  Ограничения на meta контейнер, в рамках которого запускаются контейнеры download_* и verify_*
    *  Изменение гарантий/лимитов производится "на лету" - без остановки запущенных подконтейнеров
    *  Изменение гарантий в меньшую сторону может привести к прерыванию запущенных подконтейнеров
    */
   compute_resources: TComputeResources | undefined;
   /**
    *  Максимальное количество одновременно работающих download контейнеров
    *  Если равно 0, то будет интерпретироваться как 3
    */
   active_download_containers_limit: number;
   /**
    *  Дефолтная политика сохранения/удаления исходных файлов лейров
    */
   default_layer_source_file_storage_policy: ELayerSourceFileStoragePolicy;
   /**
    *  Максимальное количество одновременно работающих verify контейнеров
    *  Если равно 0, то будет интерпретироваться как 100
    */
   active_verify_containers_limit: number;
}

export interface TBoxStatus {
   id: string;
   state: EBoxState;
   inits: TContainerStatus[];
   /**
    *  Адрес из подсети c id 'boxes_subnet', выделенной на поде (см ip6_subnet_allocations)
    *  Пустая строка, если подсеть не выделена
    */
   ip6_address: string;
   /**
    *  Имя porto meta контейнера (относительно корневого контейнера пода)
    */
   container_name: string;
   ready: TCondition | undefined;
   in_progress: TCondition | undefined;
   failed: TCondition | undefined;
   revision: number;
   /**
    *  Количество неудачных попыток собрать box
    */
   fail_counter: number;
   /**
    *  spec_timestamp до которого обновлен box
    */
   spec_timestamp: number;
   /**
    *  Список статических ресурсов box
    */
   static_resource_refs: string[];
   /**
    *  Список rootfs лейеров box
    */
   rootfs_layer_refs: string[];
   /**
    *  Список вольюмов box
    */
   volume_refs: string[];
   /**
    *  Тип бокса
    *  Имеет значение "default", "system" или одно из значений в списке TProjectSpec.user_specific_box_types
    */
   specific_type: string;
}

export interface TContainerStatus {
   current: TContainerStatus_TAttemptFeedback | undefined;
   last: TContainerStatus_TAttemptFeedback | undefined;
   last_failed: TContainerStatus_TAttemptFeedback | undefined;
   zero_return_code_counter: number;
   positive_return_code_counter: number;
   oom_counter: number;
   timeout_counter: number;
   killed_externally_counter: number;
   system_failure_counter: number;
   time_limit: TTimeLimitStatus | undefined;
   /**
    *  Имя porto контейнера (относительно корневого контейнера пода)
    */
   container_name: string;
}

export interface TContainerStatus_TAttemptFeedback {
   state: EContainerState;
   return_code: number;
   /**
    *  Последние 4KB stderr
    */
   stderr_tail: Uint8Array;
   /**
    *  Последние 4KB stdout
    */
   stdout_tail: Uint8Array;
   /**
    *  Неструктурированное описание последней ошибки связанной c запуском контейнера
    */
   fail_reason: string;
   /**
    *  timestamp начала попытки
    */
   start_time: Date | { nanos: number; seconds: number } | undefined;
   /**
    *  timestamp конца попытки
    */
   death_time: Date | { nanos: number; seconds: number } | undefined;
}

export interface TCondition {
   /**
    *  True, False, Unknown
    */
   status: EConditionStatus;
   /**
    *  Одно слово, которое описывает причину перехода в этот статус (например ProcessCrash)
    */
   reason: string;
   /**
    *  Human readable причина перехода в этот статус
    */
   message: string;
   /**
    *  Время последнего обновления
    */
   last_transition_time: Date | { nanos: number; seconds: number } | undefined;
}

export interface TWorkloadStatus {
   id: string;
   box_ref: string;
   state: EWorkloadState;
   init: TContainerStatus[];
   start: TContainerStatus | undefined;
   stop_status: TStopStatus | undefined;
   destroy_status: TDestroyStatus | undefined;
   readiness_status: TReadinessStatus | undefined;
   liveness_status: TLivenessStatus | undefined;
   ready: TCondition | undefined;
   in_progress: TCondition | undefined;
   failed: TCondition | undefined;
   /**
    *  mutalbe target state
    */
   target_state: EWorkloadTargetState;
   revision: number;
   /**
    *  spec_timestamp до которого обновлен workload
    */
   spec_timestamp: number;
}

export interface TTcpCheckStatus {
   current: TTcpCheckStatus_TAttemptFeedback | undefined;
   last: TTcpCheckStatus_TAttemptFeedback | undefined;
   last_failed: TTcpCheckStatus_TAttemptFeedback | undefined;
   time_limit: TTimeLimitStatus | undefined;
   success_counter: number;
   /**
    *  Ошибки при попытке отправить запрос/получить на него ответ
    *  Не включает в себя количество timeout
    */
   error_counter: number;
   timeout_counter: number;
}

export interface TTcpCheckStatus_TAttemptFeedback {
   state: ETcpCheckState;
   fail_reason: string;
   start_time: Date | { nanos: number; seconds: number } | undefined;
   death_time: Date | { nanos: number; seconds: number } | undefined;
}

export interface THttpGetStatus {
   current: THttpGetStatus_TAttemptFeedback | undefined;
   last: THttpGetStatus_TAttemptFeedback | undefined;
   last_failed: THttpGetStatus_TAttemptFeedback | undefined;
   /**
    *  Deprecated
    */
   requests_count: number;
   time_limit: TTimeLimitStatus | undefined;
   success_counter: number;
   /**
    *  Ошибки при попытке отправить запрос/получить на него ответ
    *  В том числе ситуации, когда код ответа не лежит в промежутке от 200 до 299
    *  Не включает в себя количество timeout и wrong_answer
    */
   error_counter: number;
   timeout_counter: number;
   wrong_answer_counter: number;
}

export interface THttpGetStatus_TAttemptFeedback {
   state: EHttpGetState;
   fail_reason: string;
   inner_fail_reason: string;
   start_time: Date | { nanos: number; seconds: number } | undefined;
   death_time: Date | { nanos: number; seconds: number } | undefined;
}

export interface TUnixSignalStatus {
   current: TUnixSignalStatus_TAttemptFeedback | undefined;
   last: TUnixSignalStatus_TAttemptFeedback | undefined;
   last_failed: TUnixSignalStatus_TAttemptFeedback | undefined;
   time_limit: TTimeLimitStatus | undefined;
   success_counter: number;
   error_counter: number;
}

export interface TUnixSignalStatus_TAttemptFeedback {
   state: EUnixSignalState;
   fail_reason: string;
   send_time: Date | { nanos: number; seconds: number } | undefined;
}

export interface TStopStatus {
   container_status: TContainerStatus | undefined;
   http_get_status: THttpGetStatus | undefined;
   unix_signal_status: TUnixSignalStatus | undefined;
   /**
    *  Указан ли в спеке stop_policy
    */
   has_stop: boolean;
}

export interface TDestroyStatus {
   container_status: TContainerStatus | undefined;
   http_get_status: THttpGetStatus | undefined;
   /**
    *  Указан ли в спеке destroy_policy
    */
   has_destroy: boolean;
}

export interface TReadinessStatus {
   container_status: TContainerStatus | undefined;
   tcp_check_status: TTcpCheckStatus | undefined;
   http_get_status: THttpGetStatus | undefined;
   /**
    *  Указан ли в спеке readiness_check
    */
   has_readiness: boolean;
}

export interface TLivenessStatus {
   container_status: TContainerStatus | undefined;
   tcp_check_status: TTcpCheckStatus | undefined;
   http_get_status: THttpGetStatus | undefined;
   /**
    *  Указан ли в спеке liveness_check
    */
   has_liveness: boolean;
}

export interface TDownloadProgress {
   /**
    *  Прогресс скачивания в процентах
    */
   percent: number;
}

export interface TLayerStatus {
   id: string;
   state: ELayerState;
   /**
    *  Количество попыток скачать ресурс
    *  Нет гарантии монотонно растущих счётчиков, т.к. их значения могут быть удалены на хосте
    */
   download_attempts_counter: number;
   /**
    *  Количество попыток проверить контрольную сумму
    */
   verification_attempts_counter: number;
   /**
    *  Общее количество ошибок связанное с ресурсом
    */
   fail_counter: number;
   ready: TCondition | undefined;
   in_progress: TCondition | undefined;
   failed: TCondition | undefined;
   revision: number;
   /**
    *  spec_timestamp до которого обновлен layer
    */
   spec_timestamp: number;
   /**
    *  Прогресс скачивания layer
    */
   download_progress: TDownloadProgress | undefined;
}

export interface TStaticResourceStatus {
   id: string;
   state: EStaticResourceState;
   /**
    *  Количество попыток скачать ресурс
    *  Нет гарантии монотонно растущих счётчиков, т.к. их значения могут быть удалены на хосте
    */
   download_attempts_counter: number;
   /**
    *  Количество попыток проверить контрольную сумму
    */
   verification_attempts_counter: number;
   /**
    *  Общее количество ошибок связанное с ресурсом
    */
   fail_counter: number;
   ready: TCondition | undefined;
   in_progress: TCondition | undefined;
   failed: TCondition | undefined;
   revision: number;
   /**
    *  spec_timestamp до которого обновлен static_resource
    */
   spec_timestamp: number;
   /**
    *  Прогресс скачивания static_resource
    */
   download_progress: TDownloadProgress | undefined;
}

export interface TResourceGangStatus {
   layers: TLayerStatus[];
   static_resources: TStaticResourceStatus[];
}

export interface TVolumeStatus {
   id: string;
   state: EVolumeState;
   /**
    *  Количество неудачных попыток собрать вольюм
    */
   fail_counter: number;
   ready: TCondition | undefined;
   in_progress: TCondition | undefined;
   failed: TCondition | undefined;
   revision: number;
   /**
    *  spec_timestamp до которого обновлен volume
    */
   spec_timestamp: number;
}
