import { useCallback, useEffect, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { EAccessControlPermission, EObjectType } from '../../proto-typings';
import { useConfig } from '../../services';
import { CheckObjectPermissionsParams } from '../../services/api/services/ypObjectServiceApiBase';
import { StoreKeysMeta, useArrayMemo, createKey } from '../../utils';
import { fetchPermissions, selectPermissions } from '../slices/yp/slices/permissions';
import type { RootState } from '../store';

const defaultPath = '/access/deploy/approvers';
const mandatoryPath = '/access/deploy/mandatory_approvers';

const getRequestData = (stageId: string) => (login: string) => (path: string): CheckObjectPermissionsParams => ({
   type: EObjectType.OT_APPROVAL_POLICY,
   permission: EAccessControlPermission.ACA_USE,
   id: stageId,
   login,
   path,
});

const getSubrequests = (stageIds: string[], logins: string[]): CheckObjectPermissionsParams[] =>
   stageIds.flatMap(stageId => {
      const getStageData = getRequestData(stageId);
      return logins.flatMap(login => [getStageData(login)(defaultPath), getStageData(login)(mandatoryPath)]);
   });

interface Params {
   stageIds: string[];
   logins: string[];
   skipLoad?: boolean;
}

export type ApprovalPermissionsResult = {
   [permissionKey: string]: boolean;
} & StoreKeysMeta<'stageId' | 'login' | 'type'>;

export function useApprovalPermissions({ stageIds, logins, skipLoad = false }: Params): ApprovalPermissionsResult {
   const dispatch = useDispatch();

   const stageList = useArrayMemo(stageIds);
   const { user } = useConfig()!;
   const userLogin = user.login;

   const memoLogins = useArrayMemo(logins);
   const loginsList = useMemo(() => [...new Set([...memoLogins, userLogin])], [memoLogins, userLogin]);

   const permissions = useSelector((state: RootState) => selectPermissions(state));

   const existPermissions = useMemo(() => {
      for (const login of loginsList) {
         for (const stageId of stageList) {
            const data = (permissions[login]?.approval_policy || {})[stageId];
            if (!data?.use) {
               return false;
            }
         }
      }
      return true;
   }, [loginsList, permissions, stageList]);

   useEffect(() => {
      if (!skipLoad && !existPermissions) {
         dispatch(fetchPermissions(getSubrequests(stageList, loginsList)));
      }
   }, [dispatch, existPermissions, loginsList, skipLoad, stageList]);

   const getCurrentPermissions = useCallback(
      (login: string, stageId: string, path: string) =>
         (permissions[login]?.approval_policy || {})[stageId]?.use?.[path],
      [permissions],
   );

   const loginsPermissions = useMemo(() => {
      const result: ApprovalPermissionsResult = {};
      for (const stageId of stageList) {
         for (const login of loginsList) {
            const existDefault = getCurrentPermissions(login, stageId, defaultPath) ?? false;
            const existMandatory = getCurrentPermissions(login, stageId, mandatoryPath) ?? false;
            result[createKey({ stageId, login, type: 'default' })] = existDefault;
            result[createKey({ stageId, login, type: 'mandatory' })] = existMandatory;

            if (login === userLogin) {
               result[createKey({ stageId, login: '', type: 'default' })] = existDefault;
               result[createKey({ stageId, login: '', type: 'mandatory' })] = existMandatory;
            }
         }
      }

      return result;
   }, [stageList, loginsList, getCurrentPermissions, userLogin]);

   return loginsPermissions;
}
