import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { ApprovalPolicyRequestKeys, ApprovalPolicyRequestKeysActions, ApprovalPolicySpec } from '../../models/ui';
import { TApprovalPolicy, TApprovalPolicySpec } from '../../proto-typings';

import { fetchApprovalPolicy, selectApprovalPolicy } from '../slices';
import type { RootState } from '../store';
import { useNetworkErrors } from './useNetworkErrors';
import { useYpObject, YpRequestMeta } from './useYpObject';
import { ApprovalPolicyConverter } from '../../models/ui/approvalPolicy/ApprovalPolicy';
import { DeepPartial } from '../../models/typeHelpers';

interface UseApprovalPolicyResult {
   meta: YpRequestMeta;
   rawApprovalPolicy: TApprovalPolicy | null;
   approvalPolicySpec: ApprovalPolicySpec | null;
}

export function useApprovalPolicy(stageId: string, skipConverter = false) {
   const rawApprovalPolicySelector = useCallback((s: RootState) => selectApprovalPolicy(s, stageId), [stageId]);
   const rawApprovalPolicy = useSelector(rawApprovalPolicySelector);

   const dispatch = useDispatch();

   const ApprovalPolicyRequestKey = ApprovalPolicyRequestKeys.getOne({
      stageId,
      action: ApprovalPolicyRequestKeysActions.FETCH,
   });
   const networkErrors = useNetworkErrors([ApprovalPolicyRequestKey]);
   const approvalPolicyError = networkErrors[ApprovalPolicyRequestKey];

   const loadFullApprovalPolicy = useCallback(() => {
      dispatch(fetchApprovalPolicy.withRequestKey(ApprovalPolicyRequestKey)(stageId));
   }, [dispatch, ApprovalPolicyRequestKey, stageId]);

   const checkFullApprovalPolicy = useCallback((policy: DeepPartial<TApprovalPolicy>) => policy.spec !== undefined, []);

   const { meta } = useYpObject({
      object: rawApprovalPolicy,
      error: approvalPolicyError,
      loadFull: loadFullApprovalPolicy,
      checkFull: checkFullApprovalPolicy,
      key: ApprovalPolicyRequestKey,
   });

   const result: UseApprovalPolicyResult = useMemo(() => {
      const approvalPolicySpec =
         rawApprovalPolicy && !skipConverter
            ? ApprovalPolicyConverter.getSpec(rawApprovalPolicy.spec as TApprovalPolicySpec)
            : null;

      return {
         approvalPolicySpec: approvalPolicySpec ?? null,
         rawApprovalPolicy: rawApprovalPolicy as TApprovalPolicy,
         meta,
      };
   }, [meta, rawApprovalPolicy, skipConverter]);

   return result;
}
