import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { DeepPartial } from '../../models';

import { DeployTicket, DeployTicketRequestKeys } from '../../models/ui';
import { fetchDeployTicket, selectDeployTicket } from '../slices';
import type { RootState } from '../store';
import { useNetworkErrors } from './useNetworkErrors';
import { useYpObject, YpRequestMeta } from './useYpObject';
import { useNetworkRequests } from './useNetworkRequests';
import { RequestState } from '../slices/network/model';

interface UseDeployTicketResult {
   meta: YpRequestMeta;
   // rawDeployTicket: TDeployTicket | null;
   deployTicket: DeployTicket | null;
}

/**
 * Пытается взять уже загруженный деплойный тикет из хранилища redux
 *
 * Если его там нет - инициирует его загрузку в него
 */
export function useDeployTicket(ticketId: string, skipLoad = false): UseDeployTicketResult {
   const rawDeployTicketSelector = useCallback((s: RootState) => selectDeployTicket(s, ticketId), [ticketId]);
   const rawDeployTicket = useSelector(rawDeployTicketSelector);

   const dispatch = useDispatch();

   const deployTicketRequestKey = DeployTicketRequestKeys.getOne({ id: ticketId });
   const networkErrors = useNetworkErrors([deployTicketRequestKey]);
   const deployTicketError = networkErrors[deployTicketRequestKey];

   const networkRequests = useNetworkRequests([deployTicketRequestKey]);
   const { state } = networkRequests[deployTicketRequestKey] ?? {};

   const loadFullDeployTicket = useCallback(() => {
      dispatch(fetchDeployTicket.withRequestKey(deployTicketRequestKey)(ticketId));
   }, [dispatch, ticketId, deployTicketRequestKey]);

   const checkFullDeployTicket = useCallback(
      (ticket: DeepPartial<DeployTicket>) => ticket.creationDate !== undefined,
      [],
   );

   const { meta } = useYpObject({
      object: rawDeployTicket,
      error: deployTicketError,
      loadFull: loadFullDeployTicket,
      checkFull: checkFullDeployTicket,
      skipLoad: state === RequestState.PENDING || skipLoad,
      key: deployTicketRequestKey,
   });

   const result: UseDeployTicketResult = useMemo(() => {
      const deployTicket = rawDeployTicket ?? null;
      // rawDeployTicket && !skipConverter ? DeployTicketConverter.fromApi(rawDeployTicket as TDeployTicket) : null;

      return {
         deployTicket,
         // rawDeployTicket: rawDeployTicket as TDeployTicket,
         meta,
      };
   }, [meta, rawDeployTicket]);

   return result;
}
