import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { DeepPartial } from '../../models';

import { Release, ReleaseRequestKeys } from '../../models/ui';
import { fetchRelease, selectRelease } from '../slices';
import type { RootState } from '../store';
import { useNetworkErrors } from './useNetworkErrors';
import { useYpObject, YpRequestMeta } from './useYpObject';

interface UseReleaseResult {
   meta: YpRequestMeta;
   release: Release | null;
}

/**
 * Пытается взять уже загруженный стейдж из хранилища redux
 *
 * Если его там нет - инициирует его загрузку в него
 */
export function useRelease(releaseId: string | null | undefined, skipConverter = false): UseReleaseResult {
   const rawReleaseSelector = useCallback((s: RootState) => (releaseId ? selectRelease(s, releaseId) : null), [
      releaseId,
   ]);
   const rawRelease = useSelector(rawReleaseSelector);

   const dispatch = useDispatch();

   const releaseRequestKey = ReleaseRequestKeys.getOne({ id: releaseId ?? '' });
   const networkErrors = useNetworkErrors([releaseRequestKey]);
   const releaseError = networkErrors[releaseRequestKey];

   const loadFullRelease = useCallback(() => {
      if (releaseId) {
         dispatch(fetchRelease.withRequestKey(releaseRequestKey)(releaseId));
      }
   }, [dispatch, releaseId, releaseRequestKey]);

   const check = useCallback((release: DeepPartial<Release>) => release.id !== undefined, []);

   const { meta } = useYpObject({
      object: rawRelease,
      error: releaseError,
      loadFull: loadFullRelease,
      checkFull: check,
      key: releaseRequestKey,
      skipLoad: !releaseId,
   });

   return useMemo(() => {
      const release = rawRelease ?? null;
      return {
         release,
         meta,
      };
   }, [meta, rawRelease]);
}
