import { useCallback, useEffect, useMemo, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { ApiErrorItemWithData, isNonExistenceYpError, NetworkYpErrorData, YpLocation } from '../../models/api';

import { ReplicaSet, ReplicaSetConverter, ReplicaSetRequestKeys, ReplicaSetType } from '../../models/ui';
import { EObjectType, TMultiClusterReplicaSet, TReplicaSet } from '../../proto-typings';
import { getReplicaSet, selectReplicaSet } from '../slices';
import type { RootState } from '../store';
import { useNetworkErrors } from './useNetworkErrors';

interface RequestMeta {
   isProcessing: boolean;
   isNotFound: boolean;
}

interface UseReplicaSetResult {
   meta: RequestMeta;
   rawReplicaSet: TReplicaSet | TMultiClusterReplicaSet | null;
   replicaSet: ReplicaSet | null;
}

interface Props {
   type: ReplicaSetType;
   id: string;
   location: YpLocation;
}

/**
 * Пытается взять уже загруженный стейдж из хранилища redux
 *
 * Если его там нет - инициирует его загрузку в него
 */
export function useReplicaSet({ type, id, location }: Props): UseReplicaSetResult {
   const [isProcessing, setIsProcessing] = useState(false);
   const [isNotFound, setIsNotFound] = useState(false);

   const rawReplicaSetSelector = useCallback((s: RootState) => selectReplicaSet(s, location, id), [id, location]);
   const rawReplicaSet = useSelector(rawReplicaSetSelector) as TReplicaSet | TMultiClusterReplicaSet | undefined;

   const dispatch = useDispatch();

   const replicaSetRequestKey = ReplicaSetRequestKeys.getOne({ id, location });
   const networkErrors = useNetworkErrors([replicaSetRequestKey]);
   const replicaSetError = networkErrors[replicaSetRequestKey];

   const loadFullReplicaSet = useCallback(() => {
      dispatch(
         getReplicaSet.withRequestKey(replicaSetRequestKey)({
            type:
               type === ReplicaSetType.MultiCluster
                  ? EObjectType.OT_MULTI_CLUSTER_REPLICA_SET
                  : EObjectType.OT_REPLICA_SET,
            id,
            location: type === ReplicaSetType.PerCluster ? location : undefined,
         }),
      );
   }, [dispatch, id, location, replicaSetRequestKey, type]);

   const existError = Boolean(replicaSetError);
   const nonExistenceYpError =
      existError &&
      isNonExistenceYpError(
         ((replicaSetError!.error?.error as unknown) as ApiErrorItemWithData<NetworkYpErrorData>) ?? null,
      );

   // Загрузка реплика сета (если его еще нет в redux-store)
   useEffect(() => {
      // первая загрузка
      if (rawReplicaSet === undefined && !existError) {
         loadFullReplicaSet();
         setIsProcessing(true);
         setIsNotFound(false);

         return;
      }

      // rs нет в yp
      if (existError && nonExistenceYpError) {
         loadFullReplicaSet(); // повторная загрузка для вновь созданных
         setIsProcessing(false);
         setIsNotFound(true);

         return;
      }

      // проверка на редкое поле, чтобы загрузить все поля
      if (rawReplicaSet && !rawReplicaSet.meta?.type && !existError) {
         loadFullReplicaSet(); // load full

         return;
      }

      setIsProcessing(false);
      setIsNotFound(false);
   }, [loadFullReplicaSet, rawReplicaSet, existError, nonExistenceYpError]);

   return useMemo(() => {
      const replicaSet = rawReplicaSet ? ReplicaSetConverter.getUIModel(rawReplicaSet ?? null, location) : null;

      return {
         replicaSet,
         rawReplicaSet: rawReplicaSet ?? null,
         meta: { isProcessing, isNotFound } as RequestMeta,
      };
   }, [isNotFound, isProcessing, location, rawReplicaSet]);
}
