import { useCallback, useEffect, useState } from 'react';
import { format } from 'date-fns';
import { useArrayMemo } from '../../utils';
import { RequestState } from '../slices/network/model';
import { useNetworkErrors } from './useNetworkErrors';
import { useNetworkRequests } from './useNetworkRequests';

export enum ActionState {
   None,
   Pending,
   Success,
   Error,
}

interface RequestHooks {
   onSuccess?(keys: string[]): void;
   onError?(keys: string[]): void;
   onFinalize?(keys: string[]): void;
}

interface RequestControl {
   activate(startAction: () => void, activateKeys?: string[]): void;
   actionStates: Map<string, ActionState>;
}

export function useRequestControl(requestKeys: string[], hooks: RequestHooks, withDebug?: boolean): RequestControl {
   const [actionStates, setActionStates] = useState(new Map(requestKeys.map(key => [key, ActionState.None])));
   const [requestTimes, setRequestTimes] = useState(new Map(requestKeys.map(key => [key, 0])));

   const keys = useArrayMemo(requestKeys);
   const networkErrors = useNetworkErrors(keys);

   const networkRequests = useNetworkRequests(keys);

   useEffect(() => {
      const errorKeys = new Set<string>();
      const successKeys = new Set<string>();
      for (const key of keys) {
         const actionError = networkErrors[key];
         const actionRequest = networkRequests[key];
         const lastRequestTime = actionRequest?.timestamps?.pending ?? 0;
         const requestTerminate = actionRequest?.state
            ? [RequestState.OK, RequestState.ERROR].includes(actionRequest.state)
            : false;

         const requestTime = requestTimes.get(key) ?? 0;
         const actionState = actionStates.get(key) ?? ActionState.None;

         if (withDebug) {
            const lastRequestDate = new Date(lastRequestTime);
            const requestTimeDate = new Date(requestTime);
            console.log({
               actionState,
               lastRequestTime: format(lastRequestDate, 'dd MMM yyyy, HH:mm:ss'),
               requestTime: format(requestTimeDate, 'dd MMM yyyy, HH:mm:ss'),
               requestTerminate,
            });
         }
         if (actionState === ActionState.Pending && lastRequestTime >= requestTime && requestTerminate) {
            if (actionError) {
               errorKeys.add(key);
            } else {
               successKeys.add(key);
            }
         }
      }
      if (errorKeys.size > 0) {
         if (hooks.onError) {
            hooks.onError([...errorKeys]);
         }
      }
      if (successKeys.size > 0) {
         if (hooks.onSuccess) {
            hooks.onSuccess([...successKeys]);
         }
      }
      if (errorKeys.size + successKeys.size > 0) {
         if (hooks.onFinalize) {
            hooks.onFinalize([...errorKeys, ...successKeys]);
         }
         setActionStates(oldMap => {
            const newMap = new Map(oldMap);
            for (const key of [...errorKeys, ...successKeys]) {
               newMap.set(key, errorKeys.has(key) ? ActionState.Success : ActionState.Error);
            }
            return newMap;
         });
      }
      // хуки могут быть любыми, они не должны вызывать эффект
      // eslint-disable-next-line react-hooks/exhaustive-deps
   }, [actionStates, keys, networkErrors, networkRequests, requestTimes, withDebug]);

   const activate = useCallback(
      (startAction: () => void, activateKeys: string[] = keys) => {
         const now = Date.now(); // важно синхронное вычисление до startAction
         setRequestTimes(oldMap => {
            const newMap = new Map(oldMap);
            for (const key of activateKeys) {
               newMap.set(key, now);
            }
            return newMap;
         });
         setActionStates(oldMap => {
            const newMap = new Map(oldMap);
            for (const key of activateKeys) {
               newMap.set(key, ActionState.Pending);
            }
            return newMap;
         });
         startAction();
      },
      [keys],
   );

   return {
      activate,
      actionStates,
   };
}
