import { useCallback, useEffect, useState } from 'react';
import { AsyncThunkAction } from '@reduxjs/toolkit';
import { useDispatch } from 'react-redux';
import { useRequestControl } from './useRequestControl';

const LIMIT = 100;

interface Item {
   key: string;
   action: AsyncThunkAction<any, any, any>;
}

export function useRequestWithLimit<T extends Item[]>(items: T, customLimit: number = LIMIT) {
   const limit = Math.min(items.length, customLimit);

   const [started, setStarted] = useState(0);
   const [left, setLeft] = useState(items.length);

   useEffect(() => {
      setLeft(items.length);
   }, [items]);

   const [pendingFlag, setPendingFlag] = useState(false);

   const [active, setActive] = useState(false);
   const start = useCallback(() => setActive(true), []);
   const stop = useCallback(() => setActive(false), []);

   const { activate } = useRequestControl(
      items.map(e => e.key),
      {
         onSuccess(keys) {
            setLeft(l => l - keys.length);
            setPendingFlag(true);
         },
         onError(keys) {
            setLeft(l => l - keys.length);
            setPendingFlag(true);
         },
      },
   );

   const dispatch = useDispatch();

   // старт <limit> потоков
   useEffect(() => {
      if (active && started === 0) {
         activate(() => {
            for (let i = 0; i < limit; i += 1) {
               dispatch(items[i].action);
            }
            setStarted(s => s + limit);
         }, items.map(e => e.key).slice(0, limit));
      }
   }, [activate, active, dispatch, items, left, limit, started]);

   useEffect(() => {
      if (active && pendingFlag) {
         setPendingFlag(false);
         // только для ситуации, когда текущие запросы завершены
         const available = items.length - started;
         const needed = Math.min(available, limit);
         const diff = started - (items.length - left);
         const pendingForRequests = Math.max(needed - Math.max(needed + diff - limit, 0), 0);
         activate(() => {
            for (let i = 0; i < pendingForRequests; i += 1) {
               dispatch(items[i + started].action);
            }
            setStarted(s => s + pendingForRequests);
         }, items.map(e => e.key).slice(started, started + pendingForRequests));
      }
   }, [activate, active, dispatch, items, left, limit, pendingFlag, started]);

   return { start, stop };
}
