import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { DeepPartial } from '../../models';

import { Stage, StageConverter, StageRequestKeys } from '../../models/ui';
import { StageStatus, StageStatusConverter, StageStatusLabel } from '../../models/ui/stage/Stage/StageStatusConverter';
import { TStage, TStageStatus } from '../../proto-typings';
import { useConfig } from '../../services';
import { getStages, selectStage } from '../slices';
import { RequestState } from '../slices/network/model';
import type { RootState } from '../store';
import { useNetworkErrors } from './useNetworkErrors';
import { useNetworkRequests } from './useNetworkRequests';
import { useYpObject, YpRequestMeta } from './useYpObject';

interface UseStageResult {
   meta: YpRequestMeta;
   rawStage: TStage | null;
   stage: Stage | null;
   stageStatus: StageStatus | null;
   stageStatusLabel: StageStatusLabel | null;
}

/**
 * Пытается взять уже загруженный стейдж из хранилища redux
 *
 * Если его там нет - инициирует его загрузку в него
 */
export function useStage(stageId: string, skipConverter = false, skipLoad = false): UseStageResult {
   const deployEngine = useConfig()!.getDeployEngine();

   const rawStageSelector = useCallback((s: RootState) => selectStage(s, stageId), [stageId]);
   const rawStage = useSelector(rawStageSelector) as DeepPartial<TStage> | undefined;

   const dispatch = useDispatch();

   const stageRequestKey = StageRequestKeys.getOne({ id: stageId });
   const networkErrors = useNetworkErrors([stageRequestKey]);
   const stageError = networkErrors[stageRequestKey];

   const networkRequests = useNetworkRequests([stageRequestKey]);
   const { state } = networkRequests[stageRequestKey] ?? {};

   const loadFullStage = useCallback(() => {
      dispatch(
         getStages.withRequestKey(stageRequestKey)({
            objectIds: [stageId],
            paths: s => [s.meta, s.labels, s.spec, s.status],
         }),
      );
   }, [dispatch, stageId, stageRequestKey]);

   // комбинация редких полей, которые мы не запрашиваем отдельно на страницах
   const check = useCallback(
      (stage: DeepPartial<TStage>) => stage.spec?.revision !== undefined && stage.meta?.uuid !== undefined,
      [],
   );

   const { meta } = useYpObject({
      object: rawStage,
      error: stageError,
      loadFull: loadFullStage,
      checkFull: check,
      skipLoad: state === RequestState.PENDING || skipLoad,
      key: stageRequestKey,
   });

   return useMemo(() => {
      const stage = rawStage && !skipConverter ? StageConverter.fromApi(rawStage as TStage) : null;
      const stageStatus =
         rawStage && !skipConverter ? StageStatusConverter.getStageStatus(rawStage.status as TStageStatus) : null;

      const stageStatusLabel =
         rawStage && stage && stageStatus && !skipConverter
            ? StageStatusConverter.getStageStatusLabel(stage, stageStatus, rawStage.labels ?? {}, {
                 deployEngine,
              })
            : null;

      const result: UseStageResult = {
         stage,
         stageStatus,
         stageStatusLabel,
         rawStage: rawStage as TStage,
         meta,
      };

      return result;
   }, [deployEngine, meta, rawStage, skipConverter]);
}
