import { useEffect } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { EObjectType } from '../../proto-typings';
import { useConfig } from '../../services';
import { fetchPermissions, selectPermissions } from '../slices';
import { useNetworkErrors } from './useNetworkErrors';

type UserStagePermission = {
   delete: boolean;
   edit: boolean;
};

const SPEC = '/spec';
const ROOT = '/';

/**
 * логика только для стейджа
 * права именно с точки зрения UI, не совпадают с acl
 */
export function useStagePermissions(stageId: string, skipLoad = false): UserStagePermission {
   const config = useConfig();
   const login = config?.user?.login;
   const allowEditStage = config?.isStageAllowEditAny();

   const requestKey = `stage-permissions-${stageId}`;

   const dispatch = useDispatch();
   const userPermission = useSelector(selectPermissions);
   const errors = useNetworkErrors([requestKey]);
   const error = errors[requestKey];

   const needRequestPermissionAccess =
      login &&
      (!userPermission[login] ||
         !userPermission[login][EObjectType.OT_STAGE] ||
         !userPermission[login][EObjectType.OT_STAGE][stageId] ||
         !userPermission[login][EObjectType.OT_STAGE][stageId].write);

   useEffect(() => {
      if (needRequestPermissionAccess && login && !skipLoad) {
         dispatch(
            fetchPermissions.withRequestKey(requestKey)([
               {
                  login,
                  type: EObjectType.OT_STAGE,
                  id: stageId,
                  path: ROOT,
               },
               {
                  login,
                  type: EObjectType.OT_STAGE,
                  id: stageId,
                  path: SPEC,
               },
            ]),
         );
      }
   }, [dispatch, login, needRequestPermissionAccess, requestKey, skipLoad, stageId]);

   if (!config || !login) {
      return {
         delete: false,
         edit: false,
      };
   }

   const allowAll = allowEditStage || error;
   if (allowAll) {
      return {
         delete: true,
         edit: true,
      };
   }

   if (needRequestPermissionAccess) {
      return {
         delete: false,
         edit: false,
      };
   }

   return {
      edit: userPermission[login][EObjectType.OT_STAGE][stageId].write[SPEC] ?? false,
      delete: userPermission[login][EObjectType.OT_STAGE][stageId].write[ROOT] ?? false,
   };
}
