import { useCallback } from 'react';
import { useSelector } from 'react-redux';

import { getStageSpecLocations, StageIssue } from '../../models/ui';
import { selectDoctorInfo } from '../slices';
import type { RootState } from '../store';
import { useStage } from './useStage';

type UseStageProblemsResult = {
   isLoading: boolean;
   issues: StageIssue[];
   issuesByDeployUnit: Record<string, StageIssue[]>;
};

export function useStageProblems(stageId: string): UseStageProblemsResult {
   const { rawStage, stage } = useStage(stageId);
   const uuid = rawStage?.meta?.uuid;
   const { revision: currentRevision } = stage ?? {};

   const problemSelector = useCallback(
      (state: RootState) => (uuid ? selectDoctorInfo(state, { id: stageId, object_type: 'yp.stage', uuid }) : null),
      [stageId, uuid],
   );

   const info = useSelector(problemSelector);

   const { revision, issues = [] } = info ?? {};
   const stageIssues = issues.map(issue => ({
      ...issue,
      locations: getStageSpecLocations(rawStage, issue.locator),
   }));

   return {
      isLoading: !info || revision !== currentRevision,
      issues: stageIssues,
      issuesByDeployUnit: stageIssues
         .filter(issue => issue.locations.deployUnitId)
         .reduce((issuesByDeployUnit, issue) => {
            const duId = issue.locations.deployUnitId;
            if (duId) {
               if (!issuesByDeployUnit.hasOwnProperty(duId)) {
                  issuesByDeployUnit[duId] = [];
               }
               issuesByDeployUnit[duId].push(issue);
            }
            return issuesByDeployUnit;
         }, {} as UseStageProblemsResult['issuesByDeployUnit']),
   };
}
