import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { YpLocation } from '../../models/api';
import { TPodSet } from '../../proto-typings';
import {
   getAccountQuotaId,
   getYpAccountPodSets,
   canLoadMoreAccountPods,
   getAccountPodSets,
   selectContinuationTokens,
} from '../slices';
import { RequestState } from '../slices/network/model';
import type { RootState } from '../store';
import { useNetworkRequests } from './useNetworkRequests';

export function useYpAccountPodSets(
   accountId: string,
   cluster: YpLocation,
   segment: string,
): {
   podSets: TPodSet[];
   canLoadMore: boolean;
   isFetching: boolean;
   onLoadMore(): void;
} {
   const quotaKey = useMemo(() => getAccountQuotaId(accountId, cluster, segment), [accountId, cluster, segment]);
   const dispatch = useDispatch();
   const requests = useNetworkRequests([quotaKey]);
   const state = requests[quotaKey]?.state;
   const isFetching = state === RequestState.PENDING;
   const podSets = useSelector((rootState: RootState) => getYpAccountPodSets(rootState, quotaKey));
   const canLoadMore = useSelector((rootState: RootState) => canLoadMoreAccountPods(rootState, quotaKey));
   const { ypQuota } = useSelector(selectContinuationTokens);
   const continuationToken = ypQuota[quotaKey];

   const loadMorePods = useCallback(() => {
      if (canLoadMore) {
         dispatch(
            getAccountPodSets.withRequestKey(quotaKey).withMeta({ accountId, segment })(
               cluster,
               `[/spec/node_segment_id]="${segment}" AND [/spec/account_id]="${accountId}"`,
               ['/meta/id', '/labels', '/status/accounting'],
               continuationToken,
               25,
            ),
         );
      }
   }, [accountId, cluster, segment, quotaKey, dispatch, continuationToken, canLoadMore]);

   return {
      isFetching,
      podSets,
      canLoadMore,
      onLoadMore: loadMorePods,
   };
}
