import { useCallback, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { YpLocation } from '../../models/api';
import { EObjectType } from '../../proto-typings';
import { selectContinuationTokens, fetchHistory, selectHistory } from '../slices';
import { RequestState } from '../slices/network/model';
import type { RootState } from '../store';
import { useNetworkRequests } from './useNetworkRequests';

export function useYpNodeHistory(
   nodeId: string | undefined,
   uuid: string | undefined,
   cluster: YpLocation,
   requestKey: string,
) {
   const dispatch = useDispatch();
   const requests = useNetworkRequests([requestKey]);
   const historyRequest = requests[requestKey];
   const status = historyRequest?.state;

   const continuationTokens = useSelector(selectContinuationTokens);
   const historyId = uuid ?? nodeId;
   const currentContinuationToken = historyId ? continuationTokens.history[historyId] : undefined;

   const [lastContinuationToken, setContinuationToken] = useState<string | undefined | null>(undefined);

   if (lastContinuationToken !== currentContinuationToken) {
      setContinuationToken(currentContinuationToken);
   }

   const canFetch = lastContinuationToken !== currentContinuationToken || !lastContinuationToken;

   const fetchNodeHistory = useCallback(
      (isFirstDownload: boolean) => {
         if (nodeId && status !== RequestState.PENDING && canFetch && currentContinuationToken !== null) {
            dispatch(
               fetchHistory.withRequestKey(requestKey).withMeta({ reset: isFirstDownload })(
                  {
                     id: nodeId,
                     type: EObjectType.OT_NODE,
                     paths: ['/status/hfsm', '/status/maintenance'],
                     limit: 100,
                     continuationToken: isFirstDownload ? undefined : currentContinuationToken,
                     uuid,
                  },
                  cluster,
               ),
            );
         }
      },
      [nodeId, cluster, dispatch, requestKey, uuid, currentContinuationToken, status, canFetch],
   );

   const historySelector = useCallback(
      (rootState: RootState) => (nodeId ? selectHistory(rootState, EObjectType.OT_NODE, nodeId) : []),
      [nodeId],
   );

   const history = useSelector(historySelector);

   return { history, fetchNodeHistory, isFetching: status === RequestState.PENDING, canFetch };
}
