import { useEffect, useMemo, useState } from 'react';
import { DeepPartial } from '../../models';
import { isNonExistenceYpError } from '../../models/api';
import { noop } from '../../utils';
import { NetworkItem } from '../slices/network/model';

export interface YpRequestMeta {
   isProcessing: boolean;
   isNotFound: boolean;
}

interface Props<T> {
   object: DeepPartial<T> | undefined | null;
   error: Partial<NetworkItem> | null;
   checkFull: (obj: DeepPartial<T>) => boolean;
   loadFull: () => void;
   skipLoad?: boolean;
   key?: string;
}

const loadingKeys: Set<string> = new Set();

/**
 * Пытается отдать уже загруженный объект
 *
 * Если его нет - инициирует его загрузку
 */
export function useYpObject<T>({
   object,
   error,
   checkFull,
   loadFull,
   skipLoad = false,
   key,
}: Props<T>): { meta: YpRequestMeta } {
   const [isProcessing, setIsProcessing] = useState(false);
   const [isNotFound, setIsNotFound] = useState(false);

   const load = useMemo(
      () =>
         skipLoad || loadingKeys.has(key ?? '')
            ? noop
            : () => {
                 key && loadingKeys.add(key);
                 loadFull();
              },
      [key, loadFull, skipLoad],
   );

   const existError = Boolean(error);
   const nonExistenceYpError = existError && isNonExistenceYpError(error?.error?.error ?? null);

   // Загрузка стейджа по его ID из роута (если его еще нет в redux-store)
   useEffect(() => {
      // первая загрузка
      if (object === undefined && !existError) {
         load();
         setIsProcessing(true);
         setIsNotFound(false);

         return;
      }

      // стейджа нет в yp
      if ((existError && nonExistenceYpError) || object === null) {
         load(); // повторная загрузка для вновь созданных
         setIsProcessing(false);
         setIsNotFound(true);

         return;
      }

      // проверка, чтобы загрузить все поля
      if (object && !checkFull(object) && !existError) {
         load(); // load full

         return;
      }

      key && loadingKeys.delete(key);
      setIsProcessing(false);
      setIsNotFound(false);
   }, [existError, nonExistenceYpError, object, checkFull, loadFull, load, key]);

   return useMemo(() => ({ meta: { isNotFound, isProcessing } }), [isNotFound, isProcessing]);
}
