import { useCallback } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { YpLocation } from '../../models/api';
import { EObjectType } from '../../proto-typings';
import { fetchHistory, selectContinuationTokens, selectHistory, selectCanFecthPodHistory } from '../slices';
import { RequestState } from '../slices/network/model';
import type { RootState } from '../store';
import { useNetworkRequests } from './useNetworkRequests';

export function useYpPodHistory(podId: string, uuid: string | undefined, cluster: YpLocation, requestKey: string) {
   const dispatch = useDispatch();
   const requests = useNetworkRequests([requestKey]);
   const historyRequest = requests[requestKey];
   const status = historyRequest?.state;

   const continuationTokens = useSelector(selectContinuationTokens);
   const canFetch = useSelector((state: RootState) => selectCanFecthPodHistory(state, podId)) ?? false;
   const historyId = uuid ?? podId;
   const currentContinuationToken = historyId ? continuationTokens.history[historyId] : undefined;

   const fetchPodHistory = useCallback(
      (isFirstDownload: boolean) => {
         if (podId && status !== RequestState.PENDING && canFetch && currentContinuationToken !== null) {
            dispatch(
               fetchHistory.withRequestKey(requestKey).withMeta({ reset: isFirstDownload })(
                  {
                     id: podId,
                     type: EObjectType.OT_POD,
                     paths: ['/status/scheduling', '/status/eviction', '/status/maintenance'],
                     limit: 100,
                     continuationToken: isFirstDownload ? undefined : currentContinuationToken,
                  },
                  cluster,
               ),
            );
         }
      },
      [podId, cluster, dispatch, requestKey, currentContinuationToken, status, canFetch],
   );

   const historySelector = useCallback(
      (rootState: RootState) => (podId ? selectHistory(rootState, EObjectType.OT_POD, podId) : []),
      [podId],
   );

   const history = useSelector(historySelector);

   return {
      history,
      fetchPodHistory,
      isFetching: status === RequestState.PENDING,
      canFetch,
   };
}
