import { useCallback } from 'react';
import { useDispatch, useSelector } from 'react-redux';

import { YpLocation } from '../../models/api';
import { DeployEngine, PodSetsFiltersParams } from '../../models/ui/yp/view';
import { TPodSet } from '../../proto-typings';
import { getClusterPodSets, selectClusterPodSets, selectContinuationTokens } from '../slices';
import { RequestState } from '../slices/network/model';
import type { RootState } from '../store';
import { useNetworkRequests } from './useNetworkRequests';

const createQuery = (filters: PodSetsFiltersParams): string | undefined => {
   const queryParts = [];
   const { podSetId, accountId, segments, query, serviceId, deployEngine } = filters;
   if (podSetId) {
      queryParts.push(`is_substr('${podSetId}', [/meta/id])`);
   }
   if (accountId) {
      queryParts.push(`is_substr('${accountId}', [/meta/account_id])`);
   }
   if (segments && segments.length) {
      const segmentsQuery = segments.map((stateItem: string) => `[/spec/node_segment_id]="${stateItem}"`).join(' OR ');

      queryParts.push(`(${segmentsQuery})`);
   }
   if (serviceId) {
      queryParts.push(
         `(([/labels/deploy_engine]="${DeployEngine.YPLite}" AND [/labels/nanny_service_id]="${serviceId}") OR is_substr('${serviceId}', [/meta/id]))`,
      );
   }
   if (deployEngine && deployEngine.length) {
      const deployEngineQueries = deployEngine.map(engine => `[/labels/deploy_engine]="${engine}"`).join(' OR ');

      queryParts.push(`(${deployEngineQueries})`);
   }
   if (query) queryParts.push(query);

   return queryParts.length ? queryParts.join(' AND ') : undefined;
};

export function useYpPodSets(
   cluster: YpLocation,
   requestKey: string,
): {
   podSets: TPodSet[];
   requestPodSets: (filters?: PodSetsFiltersParams, reset?: boolean) => void;
   isFetching: boolean;
   canFetch: boolean;
} {
   const dispatch = useDispatch();
   const requests = useNetworkRequests([requestKey]);
   const podSetsRequest = requests[requestKey];
   const status = podSetsRequest?.state;
   const isFetching = status === RequestState.PENDING;

   const podSets = useSelector((rootState: RootState) => selectClusterPodSets(rootState, cluster)) ?? [];
   const {
      ypCluster: { podSets: podSetsContinuationToken },
   } = useSelector(selectContinuationTokens);

   const canFetch = Boolean(podSetsContinuationToken);

   const requestPodSets = useCallback(
      (filters, reset) => {
         dispatch(
            getClusterPodSets.withRequestKey(requestKey).withMeta({ reset })(
               cluster,
               createQuery(filters),
               undefined,
               reset ? undefined : podSetsContinuationToken,
            ),
         );
      },
      [cluster, dispatch, requestKey, podSetsContinuationToken],
   );

   return { podSets, requestPodSets, isFetching, canFetch };
}
