import { useCallback } from 'react';
import { useDispatch, useSelector } from 'react-redux';

import { YpLocation } from '../../models/api';
import { PodsFiltersParams } from '../../models/ui/yp/view';
import { TPod } from '../../proto-typings';
import { selectContinuationTokens } from '../slices';
import { RequestState } from '../slices/network/model';
import { getClusterPods, selectClusterPods } from '../slices/yp/slices/ypCluster';
import type { RootState } from '../store';
import { useNetworkRequests } from './useNetworkRequests';

// TODO: перенести в YpApi после рефакторинга
const getFiltersString = (filters: PodsFiltersParams) => {
   const { podId, podSetId, service, podState, evictionState, schedulingState, query } = filters;

   const chunks = [];

   if (podId) {
      chunks.push(`is_substr('${podId}', [/meta/id])`);
   }

   if (podSetId) {
      chunks.push(`is_substr('${podSetId}', [/meta/pod_set_id])`);
   }

   if (service) {
      chunks.push(
         `is_substr('${service}', string([/labels/nanny_service_id])) OR is_substr('${service}', string([/labels/deploy/stage_id]))`,
      );
   }

   if (podState?.length) {
      const states = podState.map(state => `[/status/agent/state]="${state}"`).join(' OR ');
      chunks.push(states);
   }

   if (evictionState?.length) {
      const evictions = evictionState.map(state => `[/status/eviction/state]="${state}"`).join(' OR ');
      chunks.push(evictions);
   }

   if (schedulingState?.length) {
      const schedulings = schedulingState.map(state => `[/status/scheduling/state]="${state}"`).join(' OR ');
      chunks.push(schedulings);
   }

   if (query) {
      chunks.push(query);
   }

   return chunks.length ? chunks.map(chunk => `(${chunk})`).join(' AND ') : undefined;
};

const LIMIT = 50;

export function useYpPods(
   cluster: YpLocation,
   requestKey: string,
   selectorFilter?: PodsFiltersParams,
): {
   pods: TPod[];
   requestPods: (filters: PodsFiltersParams, nextPage: number, reset: boolean) => void;
   isFetching: boolean;
   canFetch: boolean;
} {
   const dispatch = useDispatch();
   const requests = useNetworkRequests([requestKey]);
   const historyRequest = requests[requestKey];
   const status = historyRequest?.state;
   const isFetching = status === RequestState.PENDING;

   const pods = useSelector((state: RootState) => selectClusterPods(state, cluster, selectorFilter));
   const {
      ypCluster: { pods: podsContinuationToken },
   } = useSelector(selectContinuationTokens);

   // const isLimitReached = pods.length && pods.length % LIMIT !== 0;
   const canFetch = Boolean(podsContinuationToken);

   const requestPods = useCallback(
      (filters: PodsFiltersParams, page = 1, reset = false) => {
         dispatch(
            getClusterPods.withRequestKey(requestKey).withMeta({ reset })({
               limit: LIMIT,
               page,
               location: cluster,
               filter: getFiltersString(filters),
            }),
         );
      },
      [cluster, dispatch, requestKey],
   );

   return { pods, requestPods, isFetching, canFetch };
}
