import { useEffect, useMemo, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { YpLocation } from '../../models/api';
import { YpAccountQuota } from '../../models/ui';
import { useConfig } from '../../services';
import { getAccounts, getUserAllowedAccounts, getYpQuotas, getYpQuotaUserAccounts } from '../slices';
import { RequestState } from '../slices/network/model';
import { useNetworkRequests } from './useNetworkRequests';

const quotaNetworkRequests = {
   allowedAccounts: 'ypQuotaUserAllowedAccounts',
   accounts: 'ypQuotaAccounts',
};

export type GetPodSetsCallback = (accountId: string, cluster: YpLocation, segment: string) => void;

export function useYpQuota(): {
   quotas: Record<string, YpAccountQuota>;
   isAccountFetching: boolean;
   requestKeys: string[];
} {
   const [accountRequestKeys, setAccountRequestKey] = useState<string[]>([]);
   const [quotaRequestKeys, setQuotaRequestKey] = useState<string[]>([]);
   const dispatch = useDispatch();
   const config = useConfig();
   const userAccounts = useSelector(getYpQuotaUserAccounts);
   const quotas = useSelector(getYpQuotas);
   const accountRequests = useNetworkRequests(accountRequestKeys);
   const isAccountFetching = useMemo(
      () =>
         accountRequestKeys.some(requestKey => {
            const request = accountRequests[requestKey];
            const status = request?.state;
            return status === RequestState.PENDING;
         }),
      [accountRequestKeys, accountRequests],
   );

   const locationsDictionary = useMemo<Set<string>>(() => {
      const result = new Set<string>();

      if (config) {
         config.clusters.forEach(cluster => {
            result.add(cluster.value);
         });
      }

      return result;
   }, [config]);

   const requestKeys = useMemo(() => accountRequestKeys.concat(quotaRequestKeys), [
      accountRequestKeys,
      quotaRequestKeys,
   ]);

   useEffect(() => {
      if (config) {
         const { user } = config;
         Array.from(locationsDictionary).forEach(location => {
            const requestKey = `${quotaNetworkRequests.allowedAccounts}_${location}`;
            dispatch(getUserAllowedAccounts.withRequestKey(requestKey)(user.login, location as YpLocation));
            setAccountRequestKey([...accountRequestKeys, requestKey]);
         });
      }
   }, []); //eslint-disable-line

   useEffect(() => {
      const accountLocationsToFetch = Array.from(locationsDictionary).filter(
         location => userAccounts[location as YpLocation],
      );

      accountLocationsToFetch.forEach(location => {
         const cluster = location as YpLocation;
         const accountIds = userAccounts[cluster];

         if (accountIds && accountIds.length) {
            const requestKey = `${quotaNetworkRequests.accounts}_${location}`;
            dispatch(getAccounts.withRequestKey(requestKey)(cluster, accountIds));
            locationsDictionary.delete(location);
            setQuotaRequestKey([...quotaRequestKeys, requestKey]);
         }
      });
   }, [userAccounts]); //eslint-disable-line

   return { quotas, isAccountFetching, requestKeys };
}
