import { isEqual, sortHandler } from '@yandex-infracloud-ui/libs';

import { addListItem } from '../../utils';

import { Entity } from './Entity';
import { EntitySet } from './EntitySet';

export enum SortDirection {
   Asc,
   Desc,
}

export interface SortConfig<T> {
   direction?: SortDirection;
   entities?: T[];
   getField: (v: T) => number;
}

export interface EntityStore<T extends Entity> {
   allIds: string[];
   byIds: EntitySet<T>;
}

export const createEmptyEntityStore = <T extends Entity>(): EntityStore<T> => ({ byIds: {}, allIds: [] });

export const createCompareFn = <T>({ direction, getField }: SortConfig<T>) => (a: T, b: T) =>
   direction === SortDirection.Desc ? sortHandler(getField(b), getField(a)) : sortHandler(getField(a), getField(b));

export const sortItemsInStore = <T extends Entity>(store: EntityStore<T>, sortConfig: SortConfig<T>): void => {
   store.allIds = Object.values(store.byIds)
      .sort(createCompareFn(sortConfig))
      .map(e => e.id);
};

export const addItemToStore = <T extends Entity>(
   store: EntityStore<T>,
   entity: T,
   sortConfig?: SortConfig<T>,
): void => {
   const existEntity = store.byIds[entity.id];
   if (isEqual(existEntity, entity)) {
      return;
   }

   store.byIds[entity.id] = entity;

   if (sortConfig) {
      sortItemsInStore(store, sortConfig);
   } else {
      store.allIds = addListItem(store.allIds, entity.id);
   }
};

export const removeItemFromStore = <T extends Entity>(store: EntityStore<T>, entityId: string): void => {
   delete store.byIds[entityId];
   store.allIds = store.allIds.filter(id => id !== entityId);
};

export const listToEntityStore = <T extends Entity>(entities: T[], sortConfig?: SortConfig<T>): EntityStore<T> => {
   const store = createEmptyEntityStore<T>();

   entities.forEach(e => addItemToStore(store, e));

   if (sortConfig) {
      sortItemsInStore(store, sortConfig);
   }

   return store;
};
