import { Entity } from './Entity';
import { SortDirection } from './EntityStore';
import { addItemToGroupedEntities, GropedEntities } from './GroupedEntities';

describe('redux|models/GroupedEntities', () => {
   interface MyEntity extends Entity {
      weight: number;
   }

   describe('addItemToGroupedEntities', () => {
      it('should add first item without sorting', () => {
         const grouped: GropedEntities<MyEntity> = {};

         addItemToGroupedEntities(grouped, 'group1', 'e1');
         expect(grouped).toEqual({
            group1: ['e1'],
         });
      });

      it('should add second item without sorting', () => {
         const grouped: GropedEntities<MyEntity> = {};

         addItemToGroupedEntities(grouped, 'group1', 'e1');
         addItemToGroupedEntities(grouped, 'group1', 'e2');
         expect(grouped).toEqual({
            group1: ['e1', 'e2'],
         });
      });

      it('should add third item without sorting', () => {
         const grouped: GropedEntities<MyEntity> = {};

         addItemToGroupedEntities(grouped, 'group1', 'e1');
         addItemToGroupedEntities(grouped, 'group1', 'e2');
         addItemToGroupedEntities(grouped, 'group2', 'e3');
         expect(grouped).toEqual({
            group1: ['e1', 'e2'],
            group2: ['e3'],
         });
      });

      it('should sort (asc) on adding with sortConfig', () => {
         const list: MyEntity[] = [
            { id: 'id1', weight: 1 },
            { id: 'id2', weight: 3 },
            { id: 'id3', weight: 2 },
         ];

         const grouped: GropedEntities<MyEntity> = {};

         for (const item of list) {
            addItemToGroupedEntities(grouped, 'group1', item.id, {
               entities: list,
               getField: v => v.weight,
               direction: SortDirection.Asc,
            });
         }

         expect(grouped).toEqual({
            group1: ['id1', 'id3', 'id2'],
         });
      });

      it('should sort (desc) on adding with sortConfig', () => {
         const list: MyEntity[] = [
            { id: 'id1', weight: 1 },
            { id: 'id2', weight: 3 },
            { id: 'id3', weight: 2 },
         ];

         const grouped: GropedEntities<MyEntity> = {};

         for (const item of list) {
            addItemToGroupedEntities(grouped, 'group1', item.id, {
               entities: list,
               getField: v => v.weight,
               direction: SortDirection.Desc,
            });
         }

         expect(grouped).toEqual({
            group1: ['id2', 'id3', 'id1'],
         });
      });
   });
});
