import { PropType } from '../../models';
import { addListItem } from '../../utils';
import { Entity } from './Entity';
import { createCompareFn, SortConfig } from './EntityStore';

export interface GropedEntities<T extends Entity> {
   [groupId: string]: PropType<T, 'id'>[];
}

export const addItemToGroupedEntities = <T extends Entity>(
   groupedEntities: GropedEntities<T>,
   groupId: string,
   entityId: string,
   sortConfig?: SortConfig<T>,
): void => {
   if (!Array.isArray(groupedEntities[groupId])) {
      groupedEntities[groupId] = [];
   }

   const newIds = addListItem(groupedEntities[groupId], entityId);
   if (newIds === groupedEntities[groupId]) {
      return;
   }

   groupedEntities[groupId] = newIds;

   if (sortConfig) {
      if (!sortConfig.entities) {
         throw new Error('expected `entities` in sortConfig');
      }

      const entityMap = new Map(sortConfig.entities.map(e => [e.id, e]));

      groupedEntities[groupId] = groupedEntities[groupId]
         .map(eId => entityMap.get(eId)!)
         .filter(Boolean)
         .sort(createCompareFn(sortConfig))
         .map(e => e.id);
   }
};

export const removeItemFromGroupedEntities = <T extends Entity>(
   groupedEntities: GropedEntities<T>,
   entityId: string,
): void => {
   for (const [groupId, ids] of Object.entries(groupedEntities)) {
      groupedEntities[groupId] = ids.filter(id => id !== entityId);
   }
};
