import { createAsyncThunk, createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';
import { firstValueFrom } from 'rxjs';

import type { RootState } from '../../../redux';
import { infraDoctorApi } from '../../../services';
import { InfoRequestQuery, InfoResponse, StatRequestQuery, StatResponse } from '../../../modules/infra-doctor';
import { createKey } from '../../../utils';

type InfraDoctorState = {
   info: {
      [key: string]: InfoResponse;
   };
   stat: {
      [key: string]: StatResponse;
   };
};

const initialState: InfraDoctorState = {
   info: {},
   stat: {},
};

type InfoOutput = {
   rsp: InfoResponse;
   query: InfoRequestQuery;
};

export const fetchInfraDoctorInfo = createAsyncThunk<InfoOutput, InfoRequestQuery>('infraDoctor/info', query =>
   firstValueFrom(infraDoctorApi.info(query)).then(rsp => ({ rsp, query })),
);

type StatOutput = {
   rsp: StatResponse;
   query: StatRequestQuery;
};

export const fetchInfraDoctorStat = createAsyncThunk<StatOutput, StatRequestQuery>('infraDoctor/stat', query =>
   firstValueFrom(infraDoctorApi.stat(query)).then(rsp => ({ rsp, query })),
);

// endregion

export const infraDoctorSlice = createSlice({
   name: 'infraDoctor',
   initialState,
   reducers: {},
   extraReducers: {
      [fetchInfraDoctorInfo.fulfilled.toString()](state, action: PayloadAction<InfoOutput>) {
         const { rsp, query } = action.payload;
         const { id, uuid, object_type } = query;
         const key = createKey({ id, uuid: uuid ?? '', type: object_type });

         state.info[key] = rsp;
      },
      [fetchInfraDoctorStat.fulfilled.toString()](state, action: PayloadAction<StatOutput>) {
         const { rsp, query } = action.payload;
         const { id, uuid, object_type } = query;
         const key = createKey({ id, uuid: uuid ?? '', type: object_type });

         state.stat[key] = rsp;
      },
   },
});

export const selectDoctorInfo = createSelector(
   (s: RootState, options: InfoRequestQuery) => options.id,
   (s: RootState, options: InfoRequestQuery) => options.object_type,
   (s: RootState, options: InfoRequestQuery) => options.uuid,
   (s: RootState) => s.infraDoctor.info,
   (id, type, uuid, info) => info[createKey({ id, type, uuid: uuid ?? '' })],
);

export const selectDoctorStat = createSelector(
   (s: RootState, options: InfoRequestQuery) => options.id,
   (s: RootState, options: InfoRequestQuery) => options.object_type,
   (s: RootState, options: InfoRequestQuery) => options.uuid,
   (s: RootState) => s.infraDoctor.stat,
   (id, type, uuid, stat) => stat[createKey({ id, type, uuid: uuid ?? '' })],
);
