import { createSlice, PayloadAction } from '@reduxjs/toolkit';

// noinspection ES6PreferShortImport (circular)
import { getSliceInitialState } from '../../utils/nestedSlice';

import { NetworkAggregatorItem } from './aggregator';
import { networkReduxNamespace } from './model';
import { errorsSlice, requestsSlice } from './slices';

const initialState = {
   errors: getSliceInitialState(errorsSlice)!,
   requests: getSliceInitialState(requestsSlice)!,
};

export type NetworkState = typeof initialState;

interface RequestBasePayload {
   requestId: string;
   requestKey: string;
}

export const networkSlice = createSlice({
   name: networkReduxNamespace,
   initialState,
   reducers: {
      remove(state, action: PayloadAction<{ requestKeys: string[] }>) {
         const { requestKeys } = action.payload;
         state.errors = errorsSlice.reducer(state.errors, errorsSlice.actions.delete({ requestKeys }));
         state.requests = requestsSlice.reducer(state.requests, requestsSlice.actions.removeByKey({ requestKeys }));
      },
      addPending(state, action: PayloadAction<RequestBasePayload>) {
         const { requestKey, requestId } = action.payload;
         state.requests = requestsSlice.reducer(
            state.requests,
            requestsSlice.actions.addPending({ requestId, requestKey }),
         );
      },
      addError(
         state,
         action: PayloadAction<RequestBasePayload & { error: { saveInStore: NetworkAggregatorItem | undefined } }>,
      ) {
         const { requestId, requestKey, error: errorData } = action.payload;
         const { saveInStore } = errorData;

         if (saveInStore) {
            const { error, request } = saveInStore;
            state.errors = errorsSlice.reducer(
               state.errors,
               errorsSlice.actions.add({
                  error,
                  requestId,
                  requestKey,
               }),
            );
            state.requests = requestsSlice.reducer(
               state.requests,
               requestsSlice.actions.addError({
                  requestId,
                  requestKey,
                  request,
               }),
            );
         }
      },
      addOk(state, action: PayloadAction<RequestBasePayload>) {
         const { requestId, requestKey } = action.payload;
         state.errors = errorsSlice.reducer(state.errors, errorsSlice.actions.delete({ requestKeys: [requestKey] }));
         state.requests = requestsSlice.reducer(state.requests, requestsSlice.actions.addOk({ requestId, requestKey }));
      },
   },
});
