import { createEntityAdapter, createSlice, PayloadAction } from '@reduxjs/toolkit';
import { sortHandler } from '@yandex-infracloud-ui/libs';
import type { RootState } from '../../..';
import { NetworkAggregatorItem } from '../aggregator';
import { networkReduxNamespace, NetworkErrorsStore, NetworkError } from '../model';

const namespace = `${networkReduxNamespace}/errors`;

export const networkErrorAdapter = createEntityAdapter<NetworkError>({
   selectId: item => item.requestId,
   sortComparer: (a, b) => sortHandler(a.requestId, b.requestId),
});

const initialState: NetworkErrorsStore = {
   data: networkErrorAdapter.getInitialState(),
};

export const errorsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      add(
         state,
         action: PayloadAction<{ requestId: string; requestKey: string; error: NetworkAggregatorItem['error'] }>,
      ) {
         const { error, requestId, requestKey } = action.payload;
         const oldKeys = Object.values(networkErrorAdapter.getSelectors().selectEntities(state.data))
            .filter(e => e && e.requestKey === requestKey)
            .map(e => e!.requestId);

         if (oldKeys.length > 0) {
            networkErrorAdapter.removeMany(state.data, oldKeys);
         }

         const errorData: NetworkError = {
            ...error,
            requestId,
            requestKey,
         };
         networkErrorAdapter.addOne(state.data, errorData);
      },
      delete(state, action: PayloadAction<{ requestKeys: string[] }>) {
         const { requestKeys } = action.payload;
         const keySet = new Set(requestKeys);
         const oldKeys = Object.values(networkErrorAdapter.getSelectors().selectEntities(state.data))
            .filter(e => e && keySet.has(e.requestKey ?? ''))
            .map(e => e!.requestId);

         if (oldKeys.length > 0) {
            networkErrorAdapter.removeMany(state.data, oldKeys);
         }
      },
   },
   extraReducers: {},
});

export const selectNetworkErrors = (state: RootState) => state[networkReduxNamespace].errors.data;
