import { createEntityAdapter, createSlice, PayloadAction, createSelector } from '@reduxjs/toolkit';
import { sortHandler } from '@yandex-infracloud-ui/libs';
import type { RootState } from '../../..';
import { networkReduxNamespace, NetworkRequest, NetworkRequestsStore, RequestState } from '../model';

const namespace = `${networkReduxNamespace}/requests`;

export const networkRequestAdapter = createEntityAdapter<NetworkRequest>({
   selectId: item => item.requestId,
   sortComparer: (a, b) => sortHandler(a.requestId, b.requestId),
});

const initialState: NetworkRequestsStore = {
   data: networkRequestAdapter.getInitialState(),
};

export const requestsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      removeByKey(state, action: PayloadAction<{ requestKeys: string[] }>) {
         const { requestKeys } = action.payload;
         const keySet = new Set(requestKeys);
         const oldKeys = Object.values(networkRequestAdapter.getSelectors().selectEntities(state.data))
            .filter(e => e && keySet.has(e.requestKey ?? ''))
            .map(e => e!.requestId);
         if (oldKeys.length > 0) {
            networkRequestAdapter.removeMany(state.data, oldKeys);
         }
      },
      addPending(state, action: PayloadAction<{ requestId: string; requestKey: string }>) {
         const { requestId, requestKey } = action.payload;
         const oldKeys = Object.values(networkRequestAdapter.getSelectors().selectEntities(state.data))
            .filter(e => e && e.requestKey === requestKey && e.state === RequestState.PENDING)
            .map(e => e!.requestId);
         if (oldKeys.length > 0) {
            networkRequestAdapter.removeMany(state.data, oldKeys);
         }
         const request: NetworkRequest = {
            requestId,
            state: RequestState.PENDING,
            timestamps: {
               [RequestState.PENDING]: Number(new Date()),
            },
            lastOkTimestamp: undefined,
            requestKey,
         };
         networkRequestAdapter.addOne(state.data, request);
      },
      addError(
         state,
         action: PayloadAction<{ requestId: string; requestKey: string; request: Partial<NetworkRequest> }>,
      ) {
         const { request, requestId, requestKey } = action.payload;
         const oldRequests = Object.values(networkRequestAdapter.getSelectors().selectEntities(state.data)).filter(
            e => e && e.requestKey === requestKey && e.state !== RequestState.PENDING,
         );
         const lastOkRequest = oldRequests
            .filter(e => e?.state === RequestState.OK)
            .reduce((last, oldRequest) => {
               if (!oldRequest) {
                  return last;
               }
               if (!last) {
                  return oldRequest;
               }
               if (oldRequest.timestamps[RequestState.OK]! > last.timestamps[RequestState.OK]!) {
                  return oldRequest;
               }
               return last;
            }, undefined as NetworkRequest | undefined);

         const oldKeys = oldRequests.filter(e => e?.requestId !== lastOkRequest?.requestId).map(e => e!.requestId);
         if (oldKeys.length > 0) {
            networkRequestAdapter.removeMany(state.data, oldKeys);
         }

         const currentRequest = networkRequestAdapter.getSelectors().selectById(state.data, requestId);

         if (!currentRequest) {
            // был удален следующими периодическими запросами из состояния pending
            return;
         }
         const timestamps = {
            [RequestState.ERROR]: Number(new Date()),
         };

         if (Object.keys(currentRequest.timestamps).length > 0) {
            Object.assign(timestamps, currentRequest.timestamps);
         }

         networkRequestAdapter.updateOne(state.data, {
            id: requestId,
            changes: {
               ...request,
               state: RequestState.ERROR,
               timestamps,
               lastOkTimestamp: lastOkRequest?.timestamps[RequestState.OK],
            },
         });
      },
      addOk(state, action: PayloadAction<{ requestId: string; requestKey: string }>) {
         const { requestId, requestKey } = action.payload;

         const oldKeys = Object.values(networkRequestAdapter.getSelectors().selectEntities(state.data))
            .filter(e => e && e.requestKey === requestKey && e.state !== RequestState.PENDING)
            .map(e => e!.requestId);
         if (oldKeys.length > 0) {
            networkRequestAdapter.removeMany(state.data, oldKeys);
         }

         const currentRequest = networkRequestAdapter.getSelectors().selectById(state.data, requestId);

         if (!currentRequest) {
            // был удален следующими периодическими запросами из состояния pending
            return;
         }

         const timestamps = {
            [RequestState.OK]: Number(new Date()),
         };

         if (Object.keys(currentRequest.timestamps ?? {}).length > 0) {
            Object.assign(timestamps, currentRequest.timestamps);
         }
         networkRequestAdapter.updateOne(state.data, {
            id: requestId,
            changes: {
               timestamps,
               state: RequestState.OK,
            },
         });
      },
   },
});

export const selectNetworkRequests = (state: RootState) => state[networkReduxNamespace].requests.data;

export const selectNetworkRequest = createSelector(
   (state: RootState) => state[networkReduxNamespace].requests.data,
   (state: RootState, requestId: string) => requestId,
   (requests, requestId: string) => networkRequestAdapter.getSelectors().selectById(requests, requestId),
);
