import { createSelector, EntityState } from '@reduxjs/toolkit';
import { memoize } from 'lodash';
import type { RootState } from '../../..';
import { networkReduxNamespace, NetworkError, NetworkItem, NetworkRequest } from '../model';
import { selectNetworkRequest, networkRequestAdapter } from './requests';
import { networkErrorAdapter } from './errors';
import { ApiServiceName } from '../../../../models/api';

export const selectNetworkError = (state: RootState, requestId: string) => {
   const errors = state[networkReduxNamespace].errors.data;
   const error = networkErrorAdapter.getSelectors().selectById(errors, requestId);
   const request = selectNetworkRequest(state, requestId);
   return { error, request };
};

const selectErrors = createSelector(
   (state: RootState) => state[networkReduxNamespace].errors.data,
   (errorData: EntityState<NetworkError<ApiServiceName>>) =>
      networkErrorAdapter.getSelectors().selectEntities(errorData),
);

export const selectNetworkErrorsByKeys = createSelector(
   selectErrors,
   (state: RootState) => state[networkReduxNamespace].requests.data,
   (state: RootState, requestKeys: string[]) => requestKeys,
   (state: RootState, requestKeys: string[]) => {
      const errors = selectErrors(state);
      const keysSet = new Set(requestKeys);
      const errorIdsMap = Object.values(errors).reduce((map, e) => {
         if (e?.requestKey && keysSet.has(e.requestKey)) {
            if (!map.has(e.requestKey)) {
               map.set(e.requestKey, e.requestId);
            }
         }
         return map;
      }, new Map<string, string>());
      return JSON.stringify(Array.from(errorIdsMap.entries()));
   },
   memoize(
      (errors, requests, requestKeys, requestIdsKey) => {
         const keysSet = new Set(requestKeys);
         const errorsMap = Object.values(errors).reduce((map, e) => {
            if (e?.requestKey && keysSet.has(e.requestKey)) {
               if (!map.has(e.requestKey)) {
                  map.set(e.requestKey, e);
               }
            }
            return map;
         }, new Map<string, NetworkError>());

         return requestKeys.reduce((acc, e) => {
            const error = errorsMap.get(e);
            acc[e] = error
               ? {
                    error,
                    request: networkRequestAdapter.getSelectors().selectById(requests, error.requestId),
                 }
               : null;
            return acc;
         }, {} as Record<string, Partial<NetworkItem> | null>);
      },
      (errors, requests, requestKeys, requestIdsKey) => requestIdsKey,
   ),
);

export const selectNetworkRequestsByKeys = createSelector(
   (state: RootState) => state[networkReduxNamespace].requests.data,
   (state: RootState, requestKeys: string[]) => requestKeys,
   (state: RootState, requestKeys: string[]) => {
      const requests = state[networkReduxNamespace].requests.data;
      const requestData = networkRequestAdapter.getSelectors().selectEntities(requests);
      const keysSet = new Set(requestKeys);
      const requestsIdsMap = Object.values(requestData).reduce((map, e) => {
         if (e?.requestKey && keysSet.has(e.requestKey)) {
            const key = `${e.requestId}:${e.state}`;
            if (map.has(e.requestKey)) {
               map.set(e.requestKey, `${map.get(e.requestKey)!}::${key}`);
            } else {
               map.set(e.requestKey, key);
            }
         }
         return map;
      }, new Map<string, string>());
      return JSON.stringify(Array.from(requestsIdsMap.entries()));
   },
   memoize(
      (requests, requestKeys, requestIdsKey) => {
         const keysSet = new Set(requestKeys);
         const requestData = networkRequestAdapter.getSelectors().selectEntities(requests);
         const requestsByKeys = Object.values(requestData).reduce((map, e) => {
            const key = e?.requestKey;
            const { pending = 0 } = e?.timestamps ?? {};
            if (key && keysSet.has(key)) {
               if (!map.has(key) || map.get(key)![0] < pending) {
                  map.set(key, [pending, e!]);
               }
            }
            return map;
         }, new Map<string, [number, NetworkRequest]>());
         return requestKeys.reduce((acc, e) => {
            if (requestsByKeys.has(e)) {
               const [, request] = requestsByKeys.get(e)!;
               acc[e] = request;
            } else {
               acc[e] = null;
            }
            return acc;
         }, {} as Record<string, NetworkRequest | null>);
      },
      (requests, requestsKeys, requestIdsKey) => requestIdsKey,
   ),
);
