import { createAsyncThunk, createEntityAdapter, createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';
import { firstValueFrom } from 'rxjs';
import { StartrekTicket } from '../../../models/ui';

import type { RootState } from '../../../redux';
import { getStarTrekApi } from '../../../services';
import { StartrekApi } from '../../../services/api/services/StartrekApi';

const ticketsAdapter = createEntityAdapter<StartrekTicket>({
   selectId: r => r.id,
});

const initialState = {
   tickets: ticketsAdapter.getInitialState(),
};

interface FetchTicketSummaryInput {
   id: string;
   startrekApi?: StartrekApi;
}

interface FetchTicketSummaryOutput {
   id: string;
   summary: string;
}

export const fetchTicketSummary = createAsyncThunk<FetchTicketSummaryOutput, FetchTicketSummaryInput>(
   'startrek/fetchTicketSummary',
   ({ startrekApi, id }) =>
      firstValueFrom((startrekApi ?? getStarTrekApi()).getTicketSummary(id)).then(summary => ({ id, summary })),
);

interface FetchTicketsDataInput {
   ids: string[];
   startrekApi?: StartrekApi;
}

interface FetchTicketsDataOutput {
   data: StartrekTicket[];
}

export const fetchTicketsData = createAsyncThunk<FetchTicketsDataOutput, FetchTicketsDataInput>(
   'startrek/fetchTicketsData',
   ({ startrekApi, ids }) =>
      firstValueFrom((startrekApi ?? getStarTrekApi()).getTicketsData(ids)).then(data => ({ data })),
);

// endregion

export const startrekSlice = createSlice({
   name: 'startrek',
   initialState,
   reducers: {},
   extraReducers: {
      [fetchTicketSummary.fulfilled.toString()](state, action: PayloadAction<FetchTicketSummaryOutput>) {
         const { id, summary } = action.payload;

         const ticket = ticketsAdapter.getSelectors().selectById(state.tickets, id);

         const newTicket = {
            ...ticket,
            id,
            summary,
         };

         ticketsAdapter.upsertOne(state.tickets, newTicket);
      },
      [fetchTicketsData.fulfilled.toString()](state, action: PayloadAction<FetchTicketsDataOutput>) {
         const { data } = action.payload;
         ticketsAdapter.upsertMany(state.tickets, data);
      },
   },
});

export const selectTicket = createSelector(
   (s: RootState, id: string) => id,
   (s: RootState) => s.startrek.tickets,
   (id, tickets) => ticketsAdapter.getSelectors().selectById(tickets, id),
);
