import { createAsyncThunk, createEntityAdapter, createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';
import { TIMES_IN_MS } from '@yandex-infracloud-ui/libs';
import { firstValueFrom } from 'rxjs';
import { getXRaySummaryType, XRayStatusSummary } from '../../../models/ui/stage/XRay';

import type { RootState } from '../../../redux';
import { getXRayApi } from '../../../services';
import { XRayApi, XRayStatusInput } from '../../../services/api/services/XRayApi';
import { createKey } from '../../../utils';

type XRayStatusInfo = XRayStatusSummary &
   XRayStatusInput & {
      lastUpdateTimestamp: number;
   };

const getXrayStatusInfoKey = (info: XRayStatusInput): string => {
   const { id, uuid, revision } = info;
   return createKey({ id, uuid, revision: String(revision) });
};

const stageXrayStatusesAdapter = createEntityAdapter<XRayStatusInfo>({
   selectId: getXrayStatusInfoKey,
});

const initialState = {
   stageXrayStatuses: stageXrayStatusesAdapter.getInitialState(),
};

interface FetchXRayStatusInput extends XRayStatusInput {
   xrayApi?: XRayApi;
}
type FetchXRayStatusOutput = XRayStatusInfo;

export const fetchXRayStatus = createAsyncThunk<FetchXRayStatusOutput, FetchXRayStatusInput>(
   'xray/fetchXRayStatus',
   ({ xrayApi, id, uuid, revision }) =>
      firstValueFrom((xrayApi ?? getXRayApi()).getXRayStatus({ id, uuid, revision })).then(status => ({
         stageHealth: status.stage_health,
         issues: status.issues,
         analysisStatus: status.analysis_status,
         summaryType: getXRaySummaryType({
            stageHealth: status.stage_health,
            issues: status.issues,
            analysisStatus: status.analysis_status,
         }),
         id,
         uuid,
         revision,
         lastUpdateTimestamp: Date.now(),
      })),
);

// endregion

/**
 * время хранения статусов в сторе — полчаса
 */
const XRayStatusStoreTtl = TIMES_IN_MS.Minute * 30;

const filterOldStatuses = (currentId: string, currentRevision: number) => (info: XRayStatusInfo): boolean => {
   const { id, revision, lastUpdateTimestamp } = info;
   if (id === currentId && Number(revision) < currentRevision) {
      // старые ревизии текущего стейджа
      return true;
   }
   if (id !== currentId) {
      const now = Date.now();
      return now - lastUpdateTimestamp > XRayStatusStoreTtl;
   }

   return false;
};

export const xraySlice = createSlice({
   name: 'xray',
   initialState,
   reducers: {},
   extraReducers: {
      [fetchXRayStatus.fulfilled.toString()](state, action: PayloadAction<FetchXRayStatusOutput>) {
         const status = action.payload;
         const { id, revision } = status;
         const allStatuses = stageXrayStatusesAdapter.getSelectors().selectAll(state.stageXrayStatuses);
         const oldStatuses = allStatuses.filter(filterOldStatuses(id, Number(revision)));

         stageXrayStatusesAdapter.removeMany(state.stageXrayStatuses, oldStatuses.map(getXrayStatusInfoKey));
         stageXrayStatusesAdapter.upsertOne(state.stageXrayStatuses, status);
      },
   },
});

export const selectXRayStatus = createSelector(
   (s: RootState) => s.xray.stageXrayStatuses,
   (s: RootState, options: XRayStatusInput) => options.id,
   (s: RootState, options: XRayStatusInput) => options.uuid,
   (s: RootState, options: XRayStatusInput) => options.revision,
   (statuses, id, uuid, revision) =>
      stageXrayStatusesAdapter.getSelectors().selectById(statuses, getXrayStatusInfoKey({ id, uuid, revision })),
);
