import { getChainFromYpPath, prepareYpPaths, UsedYpObjectTypes, YpObjects, YpPaths } from '../../../models/api';
import { DeepPartial } from '../../../models';
import { EObjectType } from '../../../proto-typings';
import { hasObjectChain } from '../../../utils';
import { YpProjectsStore, YpStagesStore } from './model';
import { YpState } from './yp';

const ypStoreFieldByType: Partial<Record<UsedYpObjectTypes, keyof YpState>> = {
   stage: 'stages',
   project: 'projects',
};

type YpStoreFields = {
   [EObjectType.OT_STAGE]: YpStagesStore;
   [EObjectType.OT_PROJECT]: YpProjectsStore;
};

interface CheckYpFieldsParams<T extends UsedYpObjectTypes> {
   type: T;
   id: string;
   paths: YpPaths<T> | (string | [string, ...symbol[]])[];
}

// передавать напрямую state нельзя, это вызовет проблемы с redux-form, внутренние механизмы которого в некоторых случаях начнут запускать бесконечный цикл сброса и регистрации некоторых полей форм
export const checkYpFields = (ypState: YpState) => <T extends keyof YpStoreFields>({
   type,
   id,
   paths,
}: CheckYpFieldsParams<T>) => {
   if (!(type in ypStoreFieldByType)) {
      throw new Error(`Available types: ${Object.keys(ypStoreFieldByType)}`);
   }

   const heap = (ypState[ypStoreFieldByType[type] as keyof YpState] as unknown) as YpStoreFields[T];
   if (!(id in heap)) {
      // такого объекта ещё нет в redux
      return false;
   }

   const obj = heap[id] as DeepPartial<YpObjects[T]> | null;
   if (obj === null) {
      // такого объекта не существует
      return false;
   }

   const stringPaths = typeof paths === 'function' ? prepareYpPaths(paths) : paths;

   // если хоть один путь отсутствует, проверка возвратит false
   for (const path of stringPaths) {
      const chain = [];
      if (Array.isArray(path)) {
         const [ypath, ...tail] = path;
         chain.push(...[...getChainFromYpPath(ypath), ...tail]);
      } else {
         chain.push(...getChainFromYpPath(path));
      }
      if (!hasObjectChain(obj, chain)) {
         return false;
      }
   }

   // все пути найдены
   return true;
};
