import { TIMES_IN_MS } from '@yandex-infracloud-ui/libs';
import { isObject, mergeWith } from 'lodash';

import { DeepPartial } from '../../../models';
import { YpLocation } from '../../../models/api';
import type { HistoryEvent, YpAccountQuota } from '../../../models/ui';
import {
   EAccessControlPermission,
   EObjectType,
   TApprovalPolicy,
   TMultiClusterReplicaSet,
   TNode,
   TNodeSegment,
   TPod,
   TPodSet,
   TProject,
   TReplicaSet,
   TResource,
   TStage,
} from '../../../proto-typings';
import { YpApi } from '../../../services';
import type { RootState } from '../../store';

export const ypReduxNamespace = 'yp';
export const selectYp = (state: RootState) => state[ypReduxNamespace];

export interface YpApiMixin {
   api?: YpApi;
}

export type NonExistentYpObject = null;

export type YpHistoryStore = {
   [objectType in EObjectType]: {
      [objectId: string]: {
         [seconds_nanos: string]: HistoryEvent;
      };
   };
};

export type YpApprovalPolicyStore = {
   [stageId: string]: DeepPartial<TApprovalPolicy> | NonExistentYpObject;
};

export type YpPermissionsStore = {
   [login: string]: {
      [objectType in EObjectType]: {
         [objectId: string]: {
            [permission in EAccessControlPermission]: {
               [ypPath: string]: boolean;
            };
         };
      };
   };
};
export interface PodFilterRequestInfo {
   hasNoPods: boolean;
   hasNextPage: boolean;
   limit: number;
   page: number;
   podIds: Set<string>;
}

export const POD_REDUX_TTL = TIMES_IN_MS.Minute * 5;
export type YpPodsStore = {
   [podSetId: string]: {
      // TODO: выпилить строки вида sas-test при переделке страницы статуса, здесь оставить только YpLocation
      [location in YpLocation | string]?: {
         pods: {
            [id: string]: DeepPartial<TPod>;
         };
         requestInfo: {
            [filter: string]: PodFilterRequestInfo;
         };
         timestamps: {
            [id: string]: number;
         };
      };
   };
};

export type YpReplicaSetsStore = {
   [location in YpLocation | string]: {
      [replicaSetId: string]: DeepPartial<TReplicaSet> | DeepPartial<TMultiClusterReplicaSet> | NonExistentYpObject;
   };
};

export type YpProjectsStore = {
   [projectId: string]: DeepPartial<TProject> | NonExistentYpObject;
};

export type YpStagesStore = {
   [stageId: string]: DeepPartial<TStage>;
};

export type YPStageNamesByProjectStore = {
   [projectId: string]: {
      [stageId: string]: boolean;
   };
};

export type YpContinuationTokensStore = {
   projects: {
      [substring: string]: {
         [path: string]: {
            token: string;
            timestamp: Date;
            terminator: boolean;
         };
      };
   };
   stages: {
      [projectId_stageId: string]: {
         [path: string]: {
            token: string;
            timestamp: Date;
            terminator: boolean;
         };
      };
   };
   history: {
      [uuid: string]: string | null;
   };
   ypCluster: {
      nodes?: string;
      pods?: string;
      podSets?: string;
      replicaSets?: string;
      multiclusterReplicaSets?: string;
   };
   ypQuota: {
      [quotaKey: string]: string | undefined;
   };
   tickets?: string;
};

export type YpAvailableObjectsStore = {
   [objectType in EObjectType]: {
      [objectId: string]: {
         [login: string]: boolean;
      };
   };
};

/**
 * quotaKey format:
 * createKey(quotaParams)
 * quotaParams = {
 *    abc: string; // deploy, infracloudui
 *    location: string; // sas, man
 *    segment: string; // dev, default
 *    resourceGroup: string; cpu, disk
 *    resource: string; // cpu, ssd, hdd
 *    metric: 'total' | 'usage';
 * }
 */
export type YpQuotaStore = {
   flatStore: {
      [quotaKey: string]: number;
   };
};

export const ypMerge = (A: any, B: any) =>
   mergeWith(A, B, (a: any, b: any) => {
      if (Array.isArray(b)) {
         // перезапись массивов
         return b;
      }

      // учёт символов
      if (isObject(b)) {
         const source = a ?? {};
         for (const key of Reflect.ownKeys(b)) {
            source[key] = (b as any)[key];
         }
         return source;
      }

      // поведение по умолчанию
      return undefined;
   });

export type YPNodeInfoStore = {
   node: TNode;
   resources: TResource[];
   pods: TPod[];
};

export type YpPodInfoStore = {
   [podId: string]: {
      pod?: TPod;
      canFecthHistory?: boolean;
      lastUpdateTimestamp?: number;
      isExist?: boolean;
   };
};

export type YPNodesInfoStore = {
   nodes: TNode[];
   segments: TNodeSegment[];
};

// TODO: переработать стейт для работы с YP. Выпилить массивы
export type YpClusterStore = {
   [location in YpLocation]: {
      pods: TPod[];
      podSets: TPodSet[];
   };
};

export type YPPodSetInfoStore = {
   podSet: TPodSet;
   pods: TPod[];
};

export type YPQuotaStore = {
   quotas: Record<string, YpAccountQuota>;
   userAccounts: Record<YpLocation, string[]>;
};

export type YPReplicaSetsStore = {
   [location in YpLocation]: {
      [replicaSetId: string]: TReplicaSet;
   };
};

export type YPFilter = {
   ids: string[];
   canLoadMore: boolean;
   hasLoaded: boolean;
};

export type YPLocationFilter = {
   replicaSets: {
      [filterExpression: string]: YPFilter;
   };
   multiclusterReplicaSets: {
      [filterExpression: string]: YPFilter;
   };
};

export type YPFiltersStore = {
   [location in YpLocation]: YPLocationFilter;
};

export type YPMultiClusterReplicaSetsStore = {
   [replicaSetId: string]: TMultiClusterReplicaSet;
};

export const defaultYPLimits: Record<keyof YPLocationFilter, number> = {
   replicaSets: 50,
   multiclusterReplicaSets: 50,
};
