import { createSelector, createSlice } from '@reduxjs/toolkit';
import { prepareYpPaths } from '../../../../models/api';

import { selectYp, YpContinuationTokensStore, ypReduxNamespace } from '../model';
import { fetchHistory } from './history';
import { getNodes } from './nodes';
import { fetchProjects } from './projects';
import { fetchStages } from './stages';
import { fetchDeployTickets } from './deployTickets';
import { loadReleasesData } from './releaseData';
import { getClusterPods, getClusterPodSets } from './ypCluster';
import { getAccountPodSets, getAccountQuotaId } from './ypQuota';
import { getClusterReplicaSets } from './ypReplicaSet';
import { getMulticlusterReplicaSets } from './ypMulticlusterReplicaSet';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'continuationTokens';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {
   projects: {},
   stages: {},
   history: {},
   ypCluster: {},
   tickets: undefined,
   ypQuota: {},
} as YpContinuationTokensStore;

export const continuationTokensSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      resetTickets(state) {
         state.tickets = undefined;
      },
   },
   extraReducers: builder => {
      builder.addCase(fetchHistory.fulfilled, (state, action) => {
         const { params, response } = action.payload;
         const { uuid, id } = params[0];
         const historyId = uuid ?? id;
         const { events = [] } = response;
         if (historyId) {
            if (events.length === 0) {
               state.history[historyId] = null;
            } else {
               state.history[historyId] = response.continuation_token;
            }
         }
      });

      builder.addCase(fetchProjects.fulfilled, (state, action) => {
         const {
            response: { continuationToken: token = '' },
            params,
            meta,
         } = action.payload;
         const { paths = [''], substring = '', continuationToken } = params[0];
         const { reset } = meta;

         if (reset) {
            state.projects = {};
         }
         if (!state.projects[substring]) {
            state.projects[substring] = {};
         }
         for (const path of prepareYpPaths(paths)) {
            state.projects[substring][path] = {
               token,
               timestamp: new Date(),
               terminator: token === continuationToken,
            };
         }
         let minTimestamp = Infinity;
         for (const { timestamp } of Object.values(state.projects[substring])) {
            if (Number(timestamp) < minTimestamp) {
               minTimestamp = Number(timestamp);
            }
         }
         state.projects[substring][(Symbol.for('min') as unknown) as string] = Object.values(
            state.projects[substring],
         ).find(({ timestamp }) => Number(timestamp) === minTimestamp)!;
      });

      builder.addCase(fetchStages.fulfilled, (state, action) => {
         const {
            response: { continuationToken: token = '' },
            params,
            meta,
         } = action.payload;
         const { project = '', substring = '', paths = [''], continuationToken = '' } = params[0];
         const { reset } = meta;
         const continuationTokens = state;
         if (reset) {
            continuationTokens.stages = {};
         }
         const searchString = `${project}:${substring}`;
         if (!continuationTokens.stages[searchString]) {
            continuationTokens.stages[searchString] = {};
         }
         for (const path of prepareYpPaths(paths)) {
            continuationTokens.stages[searchString][path] = {
               token,
               timestamp: new Date(),
               terminator: token === continuationToken,
            };
         }
         let minTimestamp = Infinity;
         for (const { timestamp } of Object.values(continuationTokens.stages[searchString])) {
            if (Number(timestamp) < minTimestamp) {
               minTimestamp = Number(timestamp);
            }
         }
         continuationTokens.stages[searchString][(Symbol.for('min') as unknown) as string] = Object.values(
            continuationTokens.stages[searchString],
         ).find(({ timestamp }) => Number(timestamp) === minTimestamp)!;
      });
      builder.addCase(fetchDeployTickets.fulfilled, (state, action) => {
         const { continuationToken } = action.payload.response;
         state.tickets = continuationToken;
      });

      builder.addCase(loadReleasesData.fulfilled, (state, action) => {
         const { tickets } = action.payload.response;
         state.tickets = tickets.continuationToken;
      });

      builder.addCase(getNodes.fulfilled, (state, action) => {
         const {
            response: { continuationToken, lastPage },
         } = action.payload;
         state.ypCluster.nodes = !lastPage && continuationToken ? continuationToken : undefined;
      });

      builder.addCase(getClusterPods.fulfilled, (state, action) => {
         const {
            response: { continuationToken },
         } = action.payload;
         state.ypCluster.pods = continuationToken;
      });

      builder.addCase(getClusterPodSets.fulfilled, (state, action) => {
         const {
            response: { continuationToken },
         } = action.payload;

         state.ypCluster.podSets = continuationToken;
      });

      builder.addCase(getAccountPodSets.fulfilled, (state, action) => {
         const {
            response: { continuationToken },
            params: [location],
            meta: { accountId, segment },
         } = action.payload;

         const quotaKey = getAccountQuotaId(accountId, location, segment);
         state.ypQuota[quotaKey] = continuationToken;
      });

      builder.addCase(getClusterReplicaSets.fulfilled, (state, action) => {
         const {
            response: { continuationToken },
         } = action.payload;

         state.ypCluster.replicaSets = continuationToken;
      });

      builder.addCase(getMulticlusterReplicaSets.fulfilled, (state, action) => {
         const {
            response: { continuationToken },
         } = action.payload;

         state.ypCluster.multiclusterReplicaSets = continuationToken;
      });
   },
});

export const selectContinuationTokens = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export const selectHistoryContinuationTokens = createSelector(selectContinuationTokens, tokens => tokens.history);
