import { createEntityAdapter, createSelector, createSlice } from '@reduxjs/toolkit';

import { sortHandler } from '@yandex-infracloud-ui/libs';
import { DeployTicket, DeployTicketConverter, DeployTicketFilters } from '../../../../models/ui';
import type { RootState } from '../../../store';
import { selectYp, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { loadReleasesData } from './releaseData';
import { TDeployTicket } from '../../../../proto-typings';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'deployTickets';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const ticketsAdapter = createEntityAdapter<DeployTicket>({
   selectId: r => r.id,
   sortComparer: (a, b) => sortHandler(b.creationDate, a.creationDate),
});

const initialState = {
   data: ticketsAdapter.getInitialState(),
};

export type DeployTicketState = typeof initialState;

export const fetchDeployTicket = ypRequestAsyncThunk(`${namespace}/fetchOne`, 'getDeployTicket');
export const fetchDeployTickets = ypRequestAsyncThunk(`${namespace}/fetchMany`, 'getDeployTickets', { reset: false });
export const doDeployTicketAction = ypRequestAsyncThunk(`${namespace}/action`, 'doTicketAction');

export const deployTicketsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(fetchDeployTicket.fulfilled, (state, action) => {
         ticketsAdapter.upsertOne(state.data, action.payload.response);
      });

      builder.addCase(fetchDeployTickets.fulfilled, (state, action) => {
         const { response, meta } = action.payload;
         const tickets = (response.values as TDeployTicket[]).map(DeployTicketConverter.fromApi);

         if (meta.reset) {
            ticketsAdapter.setAll(state.data, tickets);
         } else {
            ticketsAdapter.upsertMany(state.data, tickets);
         }
      });

      builder.addCase(loadReleasesData.fulfilled, (state, action) => {
         const { response, meta } = action.payload;
         const tickets = (response.tickets.values as TDeployTicket[]).map(DeployTicketConverter.fromApi);

         if (meta.reset) {
            ticketsAdapter.setAll(state.data, tickets);
         } else {
            ticketsAdapter.upsertMany(state.data, tickets);
         }
      });
   },
});

// region Selectors

export const selectDeployTicketStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export interface SelectDeployTicketOptions {
   stageId?: string;
   releaseId?: string;
   type?: DeployTicket['sourceType'];
   filters?: DeployTicketFilters;
}

export const selectDeployTickets = createSelector(
   (_: RootState, options: SelectDeployTicketOptions) => options.stageId,
   (_: RootState, options: SelectDeployTicketOptions) => options.releaseId,
   (_: RootState, options: SelectDeployTicketOptions) => options.type,
   selectDeployTicketStore,
   (stageId, releaseId, type, tickets) =>
      ticketsAdapter
         .getSelectors()
         .selectAll(tickets.data)
         .filter(
            ticket =>
               (!stageId || ticket.stageId === stageId) &&
               (!releaseId || ticket.releaseId === releaseId) &&
               (!type || ticket.sourceType === type),
         ),
);

export const selectDeployTicketsByIds = createSelector(
   (s: RootState) => s[ypReduxNamespace].deployTickets.data,
   (s: RootState, ids: Set<string>) => ids,
   (tickets, ids) => {
      const selectTicket = ticketsAdapter.getSelectors().selectById;
      return new Map(Array.from(ids).map(id => [id, selectTicket(tickets, id) ?? null]));
   },
);

export const selectDeployTicket = createSelector(
   (_: RootState, ticketId: string) => ticketId,
   selectDeployTicketStore,
   (ticketId, tickets) => ticketsAdapter.getSelectors().selectById(tickets.data, ticketId),
);

// endregion
