import {
   createAsyncThunk,
   createEntityAdapter,
   createSelector,
   createSlice,
   EntityState,
   PayloadAction,
} from '@reduxjs/toolkit';
import { sortHandler } from '@yandex-infracloud-ui/libs';

import type { RootState } from '../../..';
import { TStageDraft, TStageSpec } from '../../../../proto-typings';
import { ypApi } from '../../../../services/api/index';
import { selectYp, YpApiMixin, ypReduxNamespace } from '../model';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'drafts';
const namespace = getNestedSliceName(ypReduxNamespace, 'drafts');

// region Initial state

export interface Draft {
   meta: {
      id: string;
      stage_id: string;
   };
   spec: {
      revision: number;
      stage_spec: TStageSpec;
      stage_revision: number;
   };
}

export interface Drafts {
   [stage_id: string]: {
      [id: string]: Draft;
   };
}

export enum DraftUpdateType {
   FORCE = 'force',
   MERGE = 'merge',
}

interface DraftUpdateOptions {
   draft: Draft;
}

interface DraftsUpdateOptions {
   drafts: Draft[];
   type?: DraftUpdateType;
}

const draftsAdapter = createEntityAdapter<TStageDraft>({
   selectId: r => r.meta!.id,
   sortComparer: (a, b) => sortHandler(b.meta!.creation_time, a.meta!.creation_time),
});

const initialState = {
   error: null as string | null,
   isLoading: false,
   lastLoading: null as number | null,
   drafts: draftsAdapter.getInitialState(),
};

// endregion

// region Async actions

type LoadDraftInput = YpApiMixin & {
   draftId: string;
};

export const loadDraft = createAsyncThunk<TStageDraft[], LoadDraftInput>(
   `${namespace}/loadDraft`,
   ({ api = ypApi, draftId }) =>
      new Promise((resolve, reject) => {
         api.getDrafts([draftId]).subscribe(resolve, reject);
      }),
);

// endregion

function updateDraft(state: any, options: DraftUpdateOptions) {
   const { draft } = options;
   const { id, stage_id } = draft.meta;
   if (!(stage_id in state.drafts)) {
      state.drafts[stage_id] = {};
   }
   state.drafts[stage_id][id] = draft;
}

export const draftsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      setLoading(state, action: PayloadAction<boolean>) {
         state.isLoading = action.payload;
      },
      updateDraft(state, action: PayloadAction<DraftUpdateOptions>) {
         updateDraft(state, action.payload);
      },
      updateDrafts(state, action: PayloadAction<DraftsUpdateOptions>) {
         const { drafts } = action.payload;
         for (const draft of drafts) {
            updateDraft(state, { draft });
         }
      },
   },
   extraReducers: builder => {
      builder.addCase(loadDraft.pending, state => {
         state.isLoading = true;
      });

      builder.addCase(loadDraft.fulfilled, (state, action) => {
         state.isLoading = false;
         state.lastLoading = Date.now();

         draftsAdapter.upsertOne(state.drafts as EntityState<TStageDraft>, action.payload[0]);
      });

      builder.addCase(loadDraft.rejected, (state, action) => {
         state.isLoading = false;
         state.error = action.error.message ?? null;
      });
   },
});

const selectDrafts = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export const selectDraft = createSelector(
   (_: RootState, draftId: string) => draftId,
   selectDrafts,
   (draftId, drafts) => draftsAdapter.getSelectors().selectById(drafts.drafts, draftId),
);
