import { createSelector, createSlice } from '@reduxjs/toolkit';

import { UsedYpObjectTypes } from '../../../../models/api';
import type { RootState } from '../../../store';

import { selectYp, YpHistoryStore, ypMerge, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';
import { EObjectType, TStage } from '../../../../proto-typings';

const name = 'history';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YpHistoryStore;

export const fetchHistory = ypRequestAsyncThunk(`${namespace}/fetch`, 'getHistory', { reset: false });

function setSafeAccountId(obj: unknown): void {
   if (!(typeof obj === 'object' && obj !== null)) {
      return;
   }
   if ('account_id' in obj) {
      const targetObj = obj as { account_id: unknown };
      if (targetObj.account_id === null) {
         targetObj.account_id = '';
      }
   }
}
function sanitizeStage(stage: TStage | undefined): void {
   setSafeAccountId(stage?.spec);
   setSafeAccountId(stage?.meta);
}

export const historySlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(fetchHistory.fulfilled, (state, action) => {
         const { params, response, meta } = action.payload;
         const { id, type } = params[0];
         const { reset } = meta;

         const history = state;

         if (!(type in history)) {
            history[type] = {};
         }

         const historyByType = history[type];
         if (!(id in historyByType)) {
            historyByType[id] = {};
         }

         if (reset) {
            // стираем старые записи при новом запросе
            historyByType[id] = {};
         }

         const historyByObject = historyByType[id];

         const events = response.events || [];

         for (const event of events) {
            const { seconds, nanos } = event.time as { seconds: number; nanos: number };
            const timeKey = `${seconds}:${nanos}`;

            if (!(timeKey in historyByObject)) {
               historyByObject[timeKey] = {
                  result: {},
               };
            }

            const eventObject = historyByObject[timeKey];

            eventObject.time = event.time;
            eventObject.event_type = event.event_type;
            eventObject.user = event.user;
            eventObject.history_enabled_attributes = event.history_enabled_attributes;

            const result = ypMerge(eventObject.result, event.results);

            // DEPLOY-5640
            if (type === EObjectType.OT_STAGE) {
               sanitizeStage(result);
            }
            eventObject.result = result;
         }
      });
   },
});

export const selectHistoryStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});
export const selectHistory = createSelector(
   selectHistoryStore,
   (state: RootState, objectType: UsedYpObjectTypes) => objectType,
   (state: RootState, objectType: UsedYpObjectTypes, objectId: string) => objectId,
   (historyStore, objectType, id) => Object.values(historyStore[objectType]?.[id] ?? {}),
);
