import { createSelector, createSlice } from '@reduxjs/toolkit';

import type { RootState } from '../../..';
import { EAccessControlAction, EAccessControlPermission, EObjectType } from '../../../../proto-typings';
import { CheckObjectPermissionsParams } from '../../../../services/api/services/ypObjectServiceApiBase';
import { selectYp, YpPermissionsStore, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { loadReleasesData } from './releaseData';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'permissions';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YpPermissionsStore;

export const fetchPermissions = ypRequestAsyncThunk(`${namespace}/fetch`, 'getPermissions');

function patchStore(
   store: YpPermissionsStore,
   actions: (EAccessControlAction | undefined)[],
   subrequests: CheckObjectPermissionsParams[],
) {
   for (let i = 0; i < subrequests.length; i += 1) {
      const { id, login, permission = EAccessControlPermission.ACA_WRITE, type, path = '' } = subrequests[i];
      if (!store[login]) {
         store[login] = Object.create(null);
      }
      if (!store[login][type]) {
         store[login][type] = {};
      }
      if (!store[login][type][id]) {
         store[login][type][id] = Object.create(null);
      }
      if (!store[login][type][id][permission]) {
         store[login][type][id][permission] = {};
      }
      store[login][type][id][permission][path] = actions[i] === EAccessControlAction.ACA_ALLOW;
   }
}

export const permissionsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(fetchPermissions.fulfilled, (state, action) => {
         const {
            response: actions,
            params: [subrequests],
         } = action.payload;
         patchStore(state, actions, subrequests);
      });

      builder.addCase(loadReleasesData.fulfilled, (state, action) => {
         const {
            response: { approval },
         } = action.payload;
         const { permissions, subrequests, fakeRequests } = approval.permissions;
         patchStore(state, permissions, subrequests);
         patchStore(
            state,
            fakeRequests.map(() => undefined),
            fakeRequests,
         );
      });
   },
});

export const selectPermissions = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

interface SelectStageOptions {
   stageId: string;
   login: string;
   permission: EAccessControlPermission;
   path: string;
}
export const selectStagePermission = createSelector(
   selectPermissions,
   (state: RootState, options: SelectStageOptions) => options.stageId,
   (state: RootState, options: SelectStageOptions) => options.login,
   (state: RootState, options: SelectStageOptions) => options.permission,
   (state: RootState, options: SelectStageOptions) => options.path,
   (permissions, stageId, login, permission, path) =>
      permissions[login]?.[EObjectType.OT_STAGE]?.[stageId]?.[permission]?.[path] ?? false,
);
