import { createSlice, PayloadAction, createSelector } from '@reduxjs/toolkit';
import { isNonExistenceYpError } from '../../../../models/api';

import { EObjectType, TPod } from '../../../../proto-typings';
import { GetApiThunkError } from '../../../utils';

import { selectYp, YpPodInfoStore, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { fetchHistory } from './history';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';
import type { RootState } from '../../../store';

const name = 'podInfo';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YpPodInfoStore;

export const getPodInfo = ypRequestAsyncThunk(`${namespace}/getPodInfo`, 'getPod');

export const requestPodEviction = ypRequestAsyncThunk(`${namespace}/requestPodEviction`, 'requestPodEviction');

export const abortPodEviction = ypRequestAsyncThunk(`${namespace}/abortPodEviction`, 'abortPodEviction');

export const acknowledgePodEviction = ypRequestAsyncThunk(
   `${namespace}/acknowledgePodEviction`,
   'acknowledgePodEviction',
);

export const podInfoSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(getPodInfo.fulfilled, (state, action) => {
         const {
            response: { pod, timestamp },
            params,
         } = action.payload;

         const [podId] = params;
         if (pod) {
            state[podId] = {
               pod: pod as TPod,
               lastUpdateTimestamp: timestamp,
               isExist: true,
            };
         }
      });

      builder.addCase(getPodInfo.rejected, (state, action) => {
         const {
            error,
            params: [podId],
         } = (action as PayloadAction<GetApiThunkError<typeof getPodInfo>>).payload;
         const isNonExistenceError = isNonExistenceYpError(error ?? null);
         if (isNonExistenceError) {
            state[podId] = {
               canFecthHistory: true,
               isExist: false,
            };
         }
      });

      builder.addCase(requestPodEviction.fulfilled, (state, action) => {
         const { params, response } = action.payload;
         const [podId] = params;
         // почему не commit_timestamp ?
         state[podId].lastUpdateTimestamp = (response as any).timestamp;
      });

      builder.addCase(abortPodEviction.fulfilled, (state, action) => {
         const { params } = action.payload;
         const [podId] = params;
         // почему не commit_timestamp ?
         state[podId].lastUpdateTimestamp = (action.payload.response as any).timestamp;
      });

      builder.addCase(acknowledgePodEviction.fulfilled, (state, action) => {
         const { params } = action.payload;
         const [podId] = params;
         // почему не commit_timestamp ?
         state[podId].lastUpdateTimestamp = (action.payload.response as any).timestamp;
      });

      builder.addCase(fetchHistory.fulfilled, (state, action) => {
         const {
            response: { events = [], continuation_token },
            params,
         } = action.payload;

         const { type, limit, id } = params[0];
         if (type === EObjectType.OT_POD) {
            if (limit) {
               const isLastItems = !continuation_token || events.length < limit;
               state[id].canFecthHistory = !isLastItems;
            } else {
               state[id].canFecthHistory = Boolean(continuation_token);
            }
         }
      });
   },
});

export const selectPodInfoStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

const getPodId = (_: RootState, podId: string) => podId;
const selectPodData = createSelector(selectPodInfoStore, getPodId, (info, podId) => info[podId] ?? {});

export const selectPod = createSelector(selectPodData, data => data.pod);
export const selectPodTimestamp = createSelector(selectPodData, data => data.lastUpdateTimestamp);
export const selectCanFecthPodHistory = createSelector(selectPodData, data => data.canFecthHistory ?? true);
export const selectIsPodExists = createSelector(selectPodData, data => data.isExist);
