import { createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';

import { DeepPartial } from '../../../../models';
import { EObjectType, TPod } from '../../../../proto-typings';
import { GetPodsReqParams } from '../../../../services/api/services/YpApi';
import { SelectObjectResult } from '../../../../services/api/services/ypObjectServiceApiBase';
import type { RootState } from '../../../store';
import { GetApiThunkError } from '../../../utils';
import { PodFilterRequestInfo, YpPodsStore, ypReduxNamespace, selectYp, POD_REDUX_TTL } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'pods';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YpPodsStore;

export type FetchPodsOutput = GetPodsReqParams & {
   response: SelectObjectResult<EObjectType.OT_POD>;
};

export const fetchPods = ypRequestAsyncThunk(`${namespace}/fetch`, 'getPods');

function prepareState(state: YpPodsStore, podSetId: string, location: string) {
   if (!(podSetId in state)) {
      state[podSetId] = {};
   }
   const podSetData = state[podSetId]!;
   if (!(location in podSetData)) {
      podSetData[location] = { pods: {}, requestInfo: {}, timestamps: {} };
   }

   const now = Date.now();
   const currentPods = podSetData[location]!;

   for (const podId of Object.keys(currentPods.timestamps)) {
      const timestamp = currentPods.timestamps[podId];
      if (now - timestamp > POD_REDUX_TTL) {
         delete currentPods.timestamps[podId];
         delete currentPods.pods[podId];
      }
   }
}

export const podsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(fetchPods.fulfilled, (state, action) => {
         const { params, response } = action.payload;
         const { podSetId = '', location, limit, page, filter = '' } = params[0];
         const pods = response.values;

         prepareState(state as YpPodsStore, podSetId, location);
         const currentPods = state[podSetId]![location]!;

         const now = Date.now();
         for (const pod of pods) {
            const id = pod.meta?.id;
            if (id) {
               currentPods.pods[id] = pod;
               currentPods.timestamps[id] = now;
            }
         }

         currentPods.requestInfo[filter] = {
            hasNoPods: pods.length === 0,
            hasNextPage: pods.length > limit,
            page,
            limit,
            podIds: new Set(pods.map(pod => pod.meta?.id ?? '').filter(id => id !== '')),
         };
      });

      builder.addCase(fetchPods.rejected, (state, action) => {
         const { podSetId = '', location, limit, page, filter = '' } = (action as PayloadAction<
            GetApiThunkError<typeof fetchPods>
         >).payload.params[0];

         prepareState(state as YpPodsStore, podSetId, location);
         const currentPods = state[podSetId]![location]!;

         currentPods.requestInfo[filter] = {
            hasNoPods: false,
            hasNextPage: false,
            page,
            limit,
            podIds: new Set(),
         };
      });
   },
});

export const selectAllPods = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});
export const selectPodsByPodSet = (state: RootState, podSetId: string) => selectAllPods(state)[podSetId];

interface SelectPodsOptions {
   location: string;
   podSetId: string;
   filter?: string;
}

export const selectPods = createSelector(
   selectAllPods,
   (_: RootState, options: SelectPodsOptions) => options.location,
   (_: RootState, options: SelectPodsOptions) => options.podSetId,
   (_: RootState, options: SelectPodsOptions) => options.filter,
   (state, location, podSetId, filter) => {
      const currentPods = state[podSetId]?.[location];
      let podList: DeepPartial<TPod>[];
      let requestInfo: PodFilterRequestInfo | null;
      if (!currentPods) {
         podList = [];
         requestInfo = null;
      } else {
         requestInfo = filter !== undefined ? currentPods.requestInfo[filter] ?? null : null;
         const { podIds = new Set() } = requestInfo ?? {};
         podList = Object.keys(currentPods.pods)
            .filter(id => podIds.has(id))
            .map(id => currentPods.pods[id])
            .filter(Boolean);
      }
      return { pods: podList, requestInfo };
   },
);

export const selectDuPods = (state: RootState, podSetId: string) => {
   const podSetPods = selectPodsByPodSet(state, podSetId) ?? {};
   return Object.keys(podSetPods).reduce((result, cluster) => {
      result[cluster] = Object.values(podSetPods[cluster]!.pods);
      return result;
   }, {} as Record<string, DeepPartial<TPod>[]>);
};

export const selectDuPodIdList = createSelector(selectDuPods, podStore =>
   Object.keys(podStore).flatMap(location =>
      podStore?.[location].map(pod => ({
         location,
         id: pod.meta?.id ?? '',
      })),
   ),
);
