import { createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';

import { DeepPartial } from '../../../../models';
import { TProject } from '../../../../proto-typings';
import type { RootState } from '../../../store';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils';
import { selectYp, ypMerge, YpProjectsStore, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';

const name = 'projects';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YpProjectsStore;

export const getProjects = ypRequestAsyncThunk(`${namespace}/get`, 'getProjects');
export const fetchProjects = ypRequestAsyncThunk(`${namespace}/fetch`, 'selectProjects', { reset: false });
export const deleteProject = ypRequestAsyncThunk(`${namespace}/remove`, 'deleteProject');
export const createProject = ypRequestAsyncThunk(`${namespace}/post`, 'createProject');

export const projectsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      deleteProject(state, action: PayloadAction<string>) {
         delete state[action.payload];
      },
   },
   extraReducers: builder => {
      builder.addCase(fetchProjects.fulfilled, (state, action) => {
         const {
            response: { values: results = [] },
            meta,
         } = action.payload;
         const { reset } = meta;
         if (reset) {
            state = {};
         }
         for (const item of results) {
            const projectId = item.meta!.id!;
            state[projectId] = ypMerge(state[projectId] ?? {}, item);
         }
      });

      builder.addCase(getProjects.fulfilled, (state, action) => {
         const { params, response } = action.payload;
         const { ids } = params[0];
         for (let j = 0; j < ids.length; j += 1) {
            const item = response[j];
            const projectId = ids[j];
            state[projectId] = ypMerge(state[projectId] ?? {}, item);
         }
      });

      builder.addCase(deleteProject.fulfilled, (state, action) => {
         const { params } = action.payload;
         const projectId = params[0];
         delete state[projectId];
      });
   },
});

export const selectProjects = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export const selectProject: (s: RootState, projectId: string) => DeepPartial<TProject> | null = createSelector(
   (s: RootState, projectId: string) => projectId,
   selectProjects,
   (projectId, projects) => projects[projectId] ?? null,
);
