import { createSlice } from '@reduxjs/toolkit';
import { getSetDifference } from '@yandex-infracloud-ui/libs';
import { forkJoin, of } from 'rxjs';
import { switchMap } from 'rxjs/operators';

import { DeployTicket, DeployTicketAggregation, DeployTicketConverter } from '../../../../models/ui';
import { ypApi as defaultYpApi } from '../../../../services';
import { GetDeployTicketsReqOptions, YpApi } from '../../../../services/api/services/YpApi';
import { createRequestThunkGenerator } from '../../../utils/RequestThunkGenerator';
import { ypReduxNamespace } from '../model';

import { EAccessControlPermission, EObjectType, TDeployTicket } from '../../../../proto-typings';
import { CheckObjectPermissionsParams } from '../../../../services/api/services/ypObjectServiceApiBase';
import { restoreObjectFromKey } from '../../../../utils';
import { getConfig } from '../../../../services/Config';
import { getNestedSliceName } from '../../../utils/nestedSlice';

const loadReleases = (releaseIds: Set<string>, ypApi: YpApi) => (tickets: DeployTicket[]) => {
   const currentReleaseIds = DeployTicketAggregation.getReleaseIds(tickets);
   const { added } = getSetDifference(releaseIds, currentReleaseIds);
   return added.size > 0 ? ypApi.getReleases([...added.values()]) : of([]);
};

const loadApprovalPolicy = (stageIds: Set<string>, ypApi: YpApi) => (tickets: DeployTicket[]) => {
   const currentStageIds = DeployTicketAggregation.getStageIds(tickets);
   const { added } = getSetDifference(stageIds, currentStageIds);
   const policies = added.size > 0 ? ypApi.fetchApprovalPolicies([...added.values()]) : of([]);
   return forkJoin({ values: policies, ids: of(added) });
};

const defaultPath = '/access/deploy/approvers';
const mandatoryPath = '/access/deploy/mandatory_approvers';

const getRequestData = (stageId: string, login: string, path: string): CheckObjectPermissionsParams => ({
   type: EObjectType.OT_APPROVAL_POLICY,
   permission: EAccessControlPermission.ACA_USE,
   id: stageId,
   login,
   path,
});

const loadApprovalPermissions = (existIds: Set<string>, stageLogins: Set<string>, ypApi: YpApi) => (
   tickets: DeployTicket[],
) => {
   const userLogin = getConfig()?.user?.login;
   const currentStageIds = DeployTicketAggregation.getApprovalStageLogins(tickets, userLogin ? [userLogin] : []);
   const { added } = getSetDifference(stageLogins, currentStageIds);
   const fakeRequests: CheckObjectPermissionsParams[] = [];
   const subrequests = [...added.values()].flatMap(key => {
      const { stageId, login } = restoreObjectFromKey(key);
      const requests = [getRequestData(stageId, login, defaultPath), getRequestData(stageId, login, mandatoryPath)];
      if (existIds.has(stageId)) {
         return requests;
      }
      fakeRequests.push(...requests);
      return [];
   });
   const permissions = subrequests.length > 0 ? ypApi.getPermissions(subrequests) : of([]);
   return forkJoin({ permissions, subrequests: of(subrequests), fakeRequests: of(fakeRequests) });
};

interface LoadAdditionalProps {
   stageIds: Set<string>;
   releaseIds: Set<string>;
   stageLogins: Set<string>;
   ypApi?: YpApi;
}

const releaseDataRequests = {
   load: (
      arg: GetDeployTicketsReqOptions,
      { stageIds, releaseIds, stageLogins, ypApi = defaultYpApi }: LoadAdditionalProps,
   ) =>
      ypApi.getDeployTickets(arg).pipe(
         switchMap(ticketsResponse => {
            const tickets = (ticketsResponse.values as TDeployTicket[]).map(DeployTicketConverter.fromApi);
            return forkJoin({
               releases: loadReleases(releaseIds, ypApi)(tickets),
               approval: loadApprovalPolicy(
                  stageIds,
                  ypApi,
               )(tickets).pipe(
                  switchMap(policies => {
                     const existIds = new Set(
                        policies.values.map(policy => policy?.meta?.id ?? '').filter(e => e !== ''),
                     );
                     return forkJoin({
                        policies: of(policies),
                        permissions: loadApprovalPermissions(existIds, stageLogins, ypApi)(tickets),
                     });
                  }),
               ),
               tickets: of(ticketsResponse),
            });
         }),
      ),
};

const createThunk = createRequestThunkGenerator(() => releaseDataRequests);

const name = 'releaseData';
const namespace = getNestedSliceName(ypReduxNamespace, name);

export const loadReleasesData = createThunk(`${namespace}/load`, 'load', { reset: false });

// фейковый срез для единообразия
export const releaseDataSlice = createSlice({
   name: namespace,
   initialState: null,
   reducers: {},
});
