import { createEntityAdapter, createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';
import { sortHandler } from '@yandex-infracloud-ui/libs';

import { ReleaseRule } from '../../../../models/ui';
import type { RootState } from '../../../store';
import { selectYp, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

// region Initial state

const name = 'releaseRules';
const namespace = getNestedSliceName(ypReduxNamespace, name);

export const releaseRuleAdapter = createEntityAdapter<ReleaseRule>({
   selectId: r => r.id,
   sortComparer: (a, b) => sortHandler(b.creationDate, a.creationDate),
});

const initialState = {
   data: releaseRuleAdapter.getInitialState(),
};

export type ReleaseRulesState = typeof initialState;
// endregion

export const fetchReleaseRule = ypRequestAsyncThunk(`${namespace}/fetchOne`, 'getReleaseRule');
export const fetchReleaseRules = ypRequestAsyncThunk(`${namespace}/fetchMany`, 'getReleaseRules', { reset: false });

export const releaseRulesSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      setRulesForTests(state, action: PayloadAction<ReleaseRule[]>) {
         releaseRuleAdapter.upsertMany(state.data, action.payload);
      },
   },
   extraReducers: builder => {
      builder.addCase(fetchReleaseRule.fulfilled, (state, action) => {
         releaseRuleAdapter.upsertOne(state.data, action.payload.response);
      });
      builder.addCase(fetchReleaseRules.fulfilled, (state, action) => {
         const { response, meta } = action.payload;

         if (meta.reset) {
            releaseRuleAdapter.removeAll(state.data);
         }
         releaseRuleAdapter.upsertMany(state.data, response);
      });
   },
});

// region Selectors

export const selectReleaseRulesStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export const selectReleaseRules = createSelector(selectReleaseRulesStore, store => store.data);

export const selectRulesByStage = createSelector(
   (_: RootState, stageId: string) => stageId,
   selectReleaseRules,
   (stageId, rules) =>
      releaseRuleAdapter
         .getSelectors()
         .selectAll(rules)
         .filter(rule => rule.stageId === stageId),
);

export const selectRule = createSelector(
   (_: RootState, ruleId: string) => ruleId,
   selectReleaseRules,
   (ruleId, rules) => releaseRuleAdapter.getSelectors().selectById(rules, ruleId),
);

// endregion
